const path = require('path');
const nEval = require('node-eval');

/**
 * Преобразует plural-форму в новый формат
 *
 * @param {String[]} items
 * @returns {I18nPluralValue}
 */
function convertPlural(items) {
    const [one, some, many, none] = items;

    return { one, some, many, none };
}
/**
 * Преобразует переводы в новый формат
 *
 * @param {TankerKitSerpKeyset} keys
 * @returns {I18nKeyset}
 */
function convert(keys) {
    const tKeys = {};

    for (const [key, val] of Object.entries(keys)) {
        tKeys[key] = Array.isArray(val) ? convertPlural(val) : val;
    }

    return tKeys;
}

class I18nKeysetHandler {
    /**
     * Обрабатывает результат работы tanker-kit/ext/dispatchers/json2bem_serp
     *
     * @param {String} filePath - пример: src/Foo/Foo.server.tsx-i18n/ru.js
     * @param {String} fileContent - пример: module.exports = { "Foo": { ... } };
     */
    constructor(filePath, fileContent) {
        /**
         * для src/Foo/Foo.i18n/ru.ts
         *   dir: 'src/Foo'
         *   name: 'Foo'
         *   lang: 'ru'
         *   keyset: {...}
         */
        const filePathObj = path.parse(filePath);

        this.lang = filePathObj.name;

        this.dir = path.dirname(filePathObj.dir);
        this.name = path.basename(this.dir);

        this.keyset = convert(nEval(fileContent)[this.name]);
    }

    merge(otherHandler) {
        Object.assign(this.keyset, otherHandler.keyset);
    }

    getPath() {
        return path.join(this.dir, `${this.name}.i18n`, `${this.lang}.ts`);
    }

    getContent(space = 4) {
        return `export const ${this.lang} = ${JSON.stringify(this.keyset, null, space)};\n`;
    }

    getIndexPath() {
        return path.join(this.dir, `${this.name}.i18n`, 'index.ts');
    }
}

class I18nKeysetHandlerStorage {
    /**
     * Хранилище кейсетов
     *
     * путь => язык => handler
     */
    constructor() {
        this.map = new Map();
    }

    /**
     * Добавляет переводы
     *
     * @param {I18nKeysetHandler} handler
     */
    add(handler) {
        if (!this.map.has(handler.dir)) {
            this.map.set(handler.dir, new Map());
        }

        const dirMap = this.map.get(handler.dir);

        if (!dirMap.has(handler.lang)) {
            dirMap.set(handler.lang, handler);
        } else {
            const storedHandler = dirMap.get(handler.lang);

            storedHandler.merge(handler);
        }
    }

    /**
     * Возвращает структуру файлов для всех найденных переводов
     * Пример:
     * {
     *   "src/Foo/Foo.i18n/ru.ts": "export const ru = { ... };",
     *   "src/Foo/Foo.i18n/index.ts": "export * from './ru';",
     *   ...
     * }
     *
     * @returns {Object}
     */
    getFiles() {
        const files = {};

        for (const dirMap of this.map.values()) {
            const langs = dirMap.keys();
            let handler;

            for (handler of dirMap.values()) {
                // основные файлы с переводами
                files[handler.getPath()] = handler.getContent();
            }

            // index-файл
            files[handler.getIndexPath()] = I18nKeysetHandlerStorage.getIndexContent(langs);
        }

        return files;
    }

    /**
     * Возвращает контент index.ts-файла
     *
     * @param {String[]} langs
     * @returns {String}
     */
    static getIndexContent(langs) {
        let content = '';

        for (const lang of langs) {
            content += `export * from './${lang}';\n`;
        }

        return content;
    }
}

module.exports = {
    convert,
    I18nKeysetHandler,
    I18nKeysetHandlerStorage,
};

/**
 * Переводы в формате
 * tanker-kit/ext/dispatchers/json2bem_serp
 *
 * @typedef {Object} TankerKitSerpKeyset
 */

/**
 * Переводы в формате
 * https://github.yandex-team.ru/search-interfaces/frontend/tree/master/packages/i18n
 *
 * @typedef {Object} I18nKeyset
 */

/**
 * Plural-перевод в формате
 * https://github.yandex-team.ru/search-interfaces/frontend/tree/master/packages/i18n#Плюрализация
 *
 * @typedef {Object} I18nPluralValue
 *
 * @param {String} one
 * @param {String} some
 * @param {String} many
 * @param {String} none
 */
