const { assert } = require('chai');
const { convert, I18nKeysetHandler, I18nKeysetHandlerStorage } = require('./ts-keyset');

describe('ts-keyset', () => {
    describe('convert', () => {
        it('should return plural converted', () => {
            const keys = {
                '{count} оценка': [
                    '{count} оценка',
                    '{count} оценки',
                    '{count} оценок',
                    'нет оценок',
                ],
                текст: 'текст',
            };

            assert.deepEqual(convert(keys), {
                '{count} оценка': {
                    one: '{count} оценка',
                    some: '{count} оценки',
                    many: '{count} оценок',
                    none: 'нет оценок',
                },
                текст: 'текст',
            });
        });
    });

    describe('I18nKeysetHandler', () => {
        const filePath = 'src/Foo/Foo.server.tsx-i18n/ru.js';
        const fileContent = 'module.exports = { Foo: { test: "test" } };';
        let handler;

        beforeEach(() => {
            handler = new I18nKeysetHandler(filePath, fileContent);
        });

        it('should parse old format', () => {
            assert.deepEqual(handler, {
                dir: 'src/Foo',
                name: 'Foo',
                lang: 'ru',
                keyset: { test: 'test' },
            });
        });

        it('should build correct path', () => {
            assert.equal(handler.getPath(), 'src/Foo/Foo.i18n/ru.ts');
        });

        it('should build correct index path', () => {
            assert.equal(handler.getIndexPath(), 'src/Foo/Foo.i18n/index.ts');
        });

        it('should build correct module content', () => {
            assert.equal(handler.getContent(0), 'export const ru = {"test":"test"};\n');
        });

        it('should merge', () => {
            const otherHandler = { keyset: { foo: 'bar' } };

            handler.merge(otherHandler);

            assert.deepEqual(handler.keyset, {
                test: 'test',
                foo: 'bar',
            });
        });
    });

    describe('I18nKeysetHandlerStorage', () => {
        const filePath = 'src/Foo/Foo.server.tsx-i18n/ru.js';
        const fileContent = 'module.exports = { Foo: {} };';
        let storage;

        beforeEach(() => {
            storage = new I18nKeysetHandlerStorage();

            storage.add(new I18nKeysetHandler(filePath, fileContent));
        });

        it('should build correct index content', () => {
            const langs = ['ru'];
            const expected = 'export * from \'./ru\';\n';

            assert.equal(I18nKeysetHandlerStorage.getIndexContent(langs), expected);
        });

        it('should build correct files', () => {
            assert.deepEqual(storage.getFiles(), {
                'src/Foo/Foo.i18n/ru.ts': 'export const ru = {};\n',
                'src/Foo/Foo.i18n/index.ts': 'export * from \'./ru\';\n',
            });
        });
    });
});
