import { e as jsonParse, l as lib, c as createCommonjsModule } from './fetch-deps.js';

/**
 * Parses a given request/response body based on the `Content-Type` header.
 */
function parseBody(body, headers) {
    var _a;
    if (body) {
        // If the intercepted request's body has a JSON Content-Type
        // parse it into an object, otherwise leave as-is.
        const hasJsonContent = (_a = headers === null || headers === void 0 ? void 0 : headers.get('content-type')) === null || _a === void 0 ? void 0 : _a.includes('json');
        if (hasJsonContent && typeof body !== 'object') {
            return jsonParse(body) || body;
        }
        return body;
    }
    // Return whatever falsey body value is given.
    return body;
}

/**
 * Formats a mocked request for introspection in browser's console.
 */
function prepareRequest(req) {
    return Object.assign(Object.assign({}, req), { headers: req.headers.getAllHeaders() });
}

/**
 * Formats a mocked response for introspection in browser's console.
 */
function prepareResponse(res) {
    const responseHeaders = lib.listToHeaders(res.headers);
    return Object.assign(Object.assign({}, res), { 
        // Parse a response JSON body for preview in the logs
        body: parseBody(res.body, responseHeaders) });
}

function getTimestamp() {
    const now = new Date();
    return [now.getHours(), now.getMinutes(), now.getSeconds()]
        .map(String)
        .map((chunk) => chunk.slice(0, 2))
        .map((chunk) => chunk.padStart(2, '0'))
        .join(':');
}

/**
 * Returns a HEX color for a given response status code number.
 */
function getStatusCodeColor(status) {
    if (status < 300) {
        return '#69AB32';
    }
    if (status < 400) {
        return '#F0BB4B';
    }
    return '#E95F5D';
}

/**
 * Converts a string path to a Regular Expression.
 * Transforms path parameters into named RegExp groups.
 */
const pathToRegExp = (path) => {
    const pattern = path
        // Escape literal dots
        .replace(/\./g, '\\.')
        // Escape literal slashes
        .replace(/\//g, '/')
        // Escape literal question marks
        .replace(/\?/g, '\\?')
        // Ignore trailing slashes
        .replace(/\/+$/, '')
        // Replace wildcard with any zero-to-any character sequence
        .replace(/\*+/g, '.*')
        // Replace parameters with named capturing groups
        .replace(/:([^\d|^\/][a-zA-Z0-9_]*(?=(?:\/|\\.)|$))/g, (_, paramName) => `(?<${paramName}>[^\/]+?)`)
        // Allow optional trailing slash
        .concat('(\\/|$)');
    return new RegExp(pattern, 'gi');
};

/**
 * Matches a given url against a path.
 */
const match = (path, url) => {
    const expression = path instanceof RegExp ? path : pathToRegExp(path);
    const match = expression.exec(url) || false;
    // Matches in strict mode: match string should equal to input (url)
    // Otherwise loose matches will be considered truthy:
    // match('/messages/:id', '/messages/123/users') // true
    const matches = path instanceof RegExp ? !!match : !!match && match[0] === match.input;
    return {
        matches,
        params: match && matches ? match.groups || null : null,
    };
};

var getCleanUrl_1 = createCommonjsModule(function (module, exports) {
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCleanUrl = void 0;
/**
 * Removes query parameters and hashes from a given URL.
 */
function getCleanUrl(url, isAbsolute) {
    if (isAbsolute === void 0) { isAbsolute = true; }
    return [isAbsolute && url.origin, url.pathname].filter(Boolean).join('');
}
exports.getCleanUrl = getCleanUrl;

});

/**
 * Returns an absolute URL based on the given relative URL, if possible.
 * Ignores regular expressions.
 */
const getAbsoluteUrl = (mask) => {
    // Global `location` object doesn't exist in Node.
    // Relative request predicate URL cannot become absolute.
    const hasLocation = typeof location !== 'undefined';
    return typeof mask === 'string' && mask.startsWith('/')
        ? `${hasLocation ? location.origin : ''}${mask}`
        : mask;
};

/**
 * Converts a given request handler mask into a URL, if given a valid URL string.
 */
function getUrlByMask(mask) {
    /**
     * If a string mask contains an asterisk (wildcard), return it as-is.
     * Converting a URL-like path string into an actual URL is misleading.
     * @see https://github.com/mswjs/msw/issues/357
     */
    if (mask instanceof RegExp || mask.includes('*')) {
        return mask;
    }
    try {
        // Attempt to create a URL instance out of the mask string.
        // Resolve mask to an absolute URL, because even a valid relative URL
        // cannot be converted into the URL instance (required absolute URL only).
        return new URL(getAbsoluteUrl(mask));
    }
    catch (error) {
        // Otherwise, the mask is a path string.
        return mask;
    }
}

function getCleanMask(resolvedMask) {
    return resolvedMask instanceof URL
        ? getCleanUrl_1.getCleanUrl(resolvedMask)
        : resolvedMask instanceof RegExp
            ? resolvedMask
            : getAbsoluteUrl(resolvedMask);
}

/**
 * Returns the result of matching given request URL
 * against a mask.
 */
function matchRequestUrl(url, mask) {
    const resolvedMask = getUrlByMask(mask);
    const cleanMask = getCleanMask(resolvedMask);
    const cleanRequestUrl = getCleanUrl_1.getCleanUrl(url);
    return match(cleanMask, cleanRequestUrl);
}

/**
 * Return the stack trace frame of a function's invocation.
 */
function getCallFrame() {
    try {
        const inspectionError = new Error();
        inspectionError.name = 'Inspection Error';
        throw inspectionError;
    }
    catch (error) {
        const frames = error.stack.split('\n');
        // Get the first frame that doesn't reference the library's internal trace.
        // Assume that frame is the invocation frame.
        const declarationFrame = frames.slice(1).find((frame) => {
            return !/(node_modules)?\/lib\/(umd|esm)\//.test(frame);
        });
        if (!declarationFrame) {
            return;
        }
        // Extract file reference from the stack frame.
        const [, declarationPath] = declarationFrame.match(/\((.+?)\)$/) || [];
        return declarationPath;
    }
}

export { getCallFrame as a, prepareRequest as b, prepareResponse as c, getTimestamp as d, getStatusCodeColor as e, getUrlByMask as g, matchRequestUrl as m, parseBody as p };
