import { c as createCommonjsModule, s as status, a as set, d as delay, f as fetch, l as lib$1, i as isNodeProcess } from './fetch-deps.js';
import { p as parse_1 } from './xml-deps.js';
import { m as mergeRight } from './errors-deps.js';
export { i as context } from './index-deps.js';
import { g as getPublicUrlFromRequest, i as isStringEqual } from './rest-deps.js';
export { R as RESTMethods, r as rest, a as restContext } from './rest-deps.js';
import { p as parseBody } from './getCallFrame-deps.js';
export { m as matchRequestUrl } from './getCallFrame-deps.js';
export { graphql, graphqlContext } from './graphql.js';

/*! *****************************************************************************
Copyright (c) Microsoft Corporation. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License"); you may not use
this file except in compliance with the License. You may obtain a copy of the
License at http://www.apache.org/licenses/LICENSE-2.0

THIS CODE IS PROVIDED ON AN *AS IS* BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION ANY IMPLIED
WARRANTIES OR CONDITIONS OF TITLE, FITNESS FOR A PARTICULAR PURPOSE,
MERCHANTABLITY OR NON-INFRINGEMENT.

See the Apache Version 2.0 License for specific language governing permissions
and limitations under the License.
***************************************************************************** */

function __awaiter(thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
}

var until = createCommonjsModule(function (module, exports) {
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Gracefully handles a given Promise factory.
 * @example
 * cosnt [error, data] = await until(() => asyncAction())
 */
exports.until = async (promise) => {
    try {
        const data = await promise().catch((error) => {
            throw error;
        });
        return [null, data];
    }
    catch (error) {
        return [error, null];
    }
};
});

var lib = createCommonjsModule(function (module, exports) {
Object.defineProperty(exports, "__esModule", { value: true });

exports.until = until.until;
});

/**
 * Attempts to resolve a Service Worker instance from a given registration,
 * regardless of its state (active, installing, waiting).
 */
const getWorkerByRegistration = (registration, absoluteWorkerUrl, findWorker) => {
    const allStates = [
        registration.active,
        registration.installing,
        registration.waiting,
    ];
    const existingStates = allStates.filter(Boolean);
    const mockWorker = existingStates.find((worker) => {
        return findWorker(worker.scriptURL, absoluteWorkerUrl);
    });
    return mockWorker || null;
};

/**
 * Returns an absolute Service Worker URL based on the given
 * relative URL (known during the registration).
 */
function getAbsoluteWorkerUrl(relativeUrl) {
    return new URL(relativeUrl, location.origin).href;
}

/**
 * Returns an active Service Worker instance.
 * When not found, registers a new Service Worker.
 */
const getWorkerInstance = (url, options = {}, findWorker) => __awaiter(void 0, void 0, void 0, function* () {
    // Resolve the absolute Service Worker URL.
    const absoluteWorkerUrl = getAbsoluteWorkerUrl(url);
    const [, mockRegistrations] = yield lib.until(() => __awaiter(void 0, void 0, void 0, function* () {
        const registrations = yield navigator.serviceWorker.getRegistrations();
        return registrations.filter((registration) => {
            return getWorkerByRegistration(registration, absoluteWorkerUrl, findWorker);
        });
    }));
    if (!navigator.serviceWorker.controller && mockRegistrations.length > 0) {
        // Reload the page when it has associated workers, but no active controller.
        // The absence of a controller can mean either:
        // - page has no Service Worker associated with it
        // - page has been hard-reloaded and its workers won't be used until the next reload.
        // Since we've checked that there are registrations associated with this page,
        // at this point we are sure it's hard reload that falls into this clause.
        location.reload();
    }
    const [existingRegistration] = mockRegistrations;
    if (existingRegistration) {
        // When the Service Worker is registered, update it and return the reference.
        return existingRegistration.update().then(() => {
            return [
                getWorkerByRegistration(existingRegistration, absoluteWorkerUrl, findWorker),
                existingRegistration,
            ];
        });
    }
    // When the Service Worker wasn't found, register it anew and return the reference.
    const [error, instance] = yield lib.until(() => __awaiter(void 0, void 0, void 0, function* () {
        const registration = yield navigator.serviceWorker.register(url, options);
        return [
            // Compare existing worker registration by its worker URL,
            // to prevent irrelevant workers to resolve here (such as Codesandbox worker).
            getWorkerByRegistration(registration, absoluteWorkerUrl, findWorker),
            registration,
        ];
    }));
    // Handle Service Worker registration errors.
    if (error) {
        const isWorkerMissing = error.message.includes('(404)');
        // Produce a custom error message when given a non-existing Service Worker url.
        // Suggest developers to check their setup.
        if (isWorkerMissing) {
            const scopeUrl = new URL((options === null || options === void 0 ? void 0 : options.scope) || '/', location.href);
            console.error(`\
[MSW] Failed to register a Service Worker for scope ('${scopeUrl.href}') with script ('${absoluteWorkerUrl}'): Service Worker script does not exist at the given path.

Did you forget to run "npx msw init <PUBLIC_DIR>"?

Learn more about creating the Service Worker script: https://mswjs.io/docs/cli/init`);
            return null;
        }
        // Fallback error message for any other registration errors.
        console.error(`[MSW] Failed to register a Service Worker:\n\m${error.message}`);
        return null;
    }
    return instance;
});

const activateMocking = (context, options) => __awaiter(void 0, void 0, void 0, function* () {
    var _a;
    (_a = context.worker) === null || _a === void 0 ? void 0 : _a.postMessage('MOCK_ACTIVATE');
    return context.events.once('MOCKING_ENABLED').then(() => {
        if (!(options === null || options === void 0 ? void 0 : options.quiet)) {
            console.groupCollapsed('%c[MSW] Mocking enabled.', 'color:orangered;font-weight:bold;');
            console.log('%cDocumentation: %chttps://mswjs.io/docs', 'font-weight:bold', 'font-weight:normal');
            console.log('Found an issue? https://github.com/mswjs/msw/issues');
            console.groupEnd();
        }
    });
});

/**
 * Creates a communication channel between the client
 * and the Service Worker associated with the given event.
 */
const createBroadcastChannel = (event) => {
    const port = event.ports[0];
    return {
        /**
         * Sends a text message to the connected Service Worker.
         */
        send(message) {
            if (port) {
                port.postMessage(message);
            }
        },
    };
};

const defaultContext = {
    status,
    set,
    delay,
    fetch,
};

/**
 * Composes a given list of functions into a new function that
 * executes from right to left.
 */
function compose(...fns) {
    return (...args) => {
        return fns.reduceRight((leftFn, rightFn) => {
            return leftFn instanceof Promise
                ? Promise.resolve(leftFn).then(rightFn)
                : rightFn(leftFn);
        }, args[0]);
    };
}

class NetworkError extends Error {
    constructor(message) {
        super(message);
        this.name = 'NetworkError';
    }
}

const defaultResponse = {
    status: 200,
    statusText: 'OK',
    body: null,
    delay: 0,
    once: false,
};
const defaultResponseTransformers = [];
function createResponseComposition(responseOverrides, defaultTransformers = defaultResponseTransformers) {
    return (...transformers) => __awaiter(this, void 0, void 0, function* () {
        const initialResponse = Object.assign({}, defaultResponse, {
            headers: new lib$1.Headers({
                'x-powered-by': 'msw',
            }),
        }, responseOverrides);
        const resolvedTransformers = [
            ...defaultTransformers,
            ...transformers,
        ].filter(Boolean);
        const resolvedResponse = resolvedTransformers.length > 0
            ? compose(...resolvedTransformers)(initialResponse)
            : initialResponse;
        return resolvedResponse;
    });
}
const response = Object.assign(createResponseComposition(), {
    once: createResponseComposition({ once: true }),
    networkError(message) {
        throw new NetworkError(message);
    },
});

/**
 * Returns a mocked response for a given request using following request handlers.
 */
const getResponse = (req, handlers) => __awaiter(void 0, void 0, void 0, function* () {
    const relevantHandlers = handlers
        .filter((requestHandler) => {
        // Skip a handler if it has been already used for a one-time response.
        return !requestHandler.shouldSkip;
    })
        .map((requestHandler) => {
        // Parse the captured request to get additional information.
        // Make the predicate function accept all the necessary information
        // to decide on the interception.
        const parsedRequest = requestHandler.parse
            ? requestHandler.parse(req)
            : null;
        return [requestHandler, parsedRequest];
    })
        .filter(([requestHandler, parsedRequest]) => {
        return requestHandler.predicate(req, parsedRequest);
    });
    if (relevantHandlers.length == 0) {
        // Handle a scenario when a request has no relevant request handlers.
        // In that case it would be bypassed (performed as-is).
        return {
            handler: null,
            response: null,
        };
    }
    const { requestHandler, parsedRequest, mockedResponse, publicRequest, } = yield relevantHandlers.reduce((asyncAcc, [requestHandler, parsedRequest]) => __awaiter(void 0, void 0, void 0, function* () {
        // Now the reduce function is async so we need to await if response was found
        const acc = yield asyncAcc;
        // If a first not empty response was found we'll stop evaluating other requests
        if (acc.requestHandler) {
            return acc;
        }
        const { getPublicRequest, defineContext, resolver } = requestHandler;
        const publicRequest = getPublicRequest
            ? getPublicRequest(req, parsedRequest)
            : req;
        const context = defineContext
            ? defineContext(publicRequest)
            : defaultContext;
        const mockedResponse = yield resolver(publicRequest, response, context);
        if (!mockedResponse) {
            return acc;
        }
        if (mockedResponse && mockedResponse.once) {
            // When responded with a one-time response, match the relevant request handler
            // as skipped, so it cannot affect the captured requests anymore.
            requestHandler.shouldSkip = true;
        }
        return {
            requestHandler,
            parsedRequest,
            mockedResponse,
            publicRequest,
        };
    }), Promise.resolve({ mockedResponse: null }));
    // Although reducing a list of relevant request handlers, it's possible
    // that in the end there will be no handler associted with the request
    // (i.e. if relevant handlers are fall-through).
    if (!requestHandler) {
        return {
            handler: null,
            response: null,
        };
    }
    return {
        handler: requestHandler,
        response: mockedResponse,
        publicRequest,
        parsedRequest,
    };
});

function onUnhandledRequest(request, handler = 'bypass') {
    if (typeof handler === 'function') {
        handler(request);
        return;
    }
    const publicUrl = getPublicUrlFromRequest(request);
    const message = `captured a ${request.method} ${request.url} request without a corresponding request handler.

  If you wish to intercept this request, consider creating a request handler for it:

  rest.${request.method.toLowerCase()}('${publicUrl}', (req, res, ctx) => {
    return res(ctx.text('body'))
  })`;
    switch (handler) {
        case 'error': {
            throw new Error(`[MSW] Error: ${message}`);
        }
        case 'warn': {
            console.warn(`[MSW] Warning: ${message}`);
        }
        default:
            return;
    }
}

function getAllCookies() {
    return parse_1(document.cookie);
}
/**
 * Returns relevant document cookies based on the request `credentials` option.
 */
function getRequestCookies(req) {
    switch (req.credentials) {
        case 'same-origin': {
            // Return document cookies only when requested a resource
            // from the same origin as the current document.
            return location.origin === req.url.origin ? getAllCookies() : {};
        }
        case 'include': {
            // Return all document cookies.
            return getAllCookies();
        }
        default: {
            return {};
        }
    }
}

const handleRequestWith = (context, options) => {
    return (event) => __awaiter(void 0, void 0, void 0, function* () {
        const channel = createBroadcastChannel(event);
        try {
            const message = JSON.parse(event.data, function (key, value) {
                if (key === 'url') {
                    return new URL(value);
                }
                // Serialize headers
                if (key === 'headers') {
                    return new lib$1.Headers(value);
                }
                // Prevent empty fields from presering an empty value.
                // It's invalid to perform a GET request with { body: "" }
                if (
                // Check if we are parsing deeper in `event.data.payload`,
                // because this custom JSON parser is invoked for each depth level.
                this.method &&
                    isStringEqual(this.method, 'GET') &&
                    key === 'body' &&
                    value === '') {
                    return undefined;
                }
                return value;
            });
            const { type, payload: req } = message;
            // Ignore irrelevant worker message types
            if (type !== 'REQUEST') {
                return null;
            }
            // Parse the request's body based on the "Content-Type" header.
            req.body = parseBody(req.body, req.headers);
            // Set document cookies on the request.
            req.cookies = getRequestCookies(req);
            const { response, handler, publicRequest, parsedRequest, } = yield getResponse(req, context.requestHandlers);
            // Handle a scenario when there is no request handler
            // found for a given request.
            if (!handler) {
                onUnhandledRequest(req, options.onUnhandledRequest);
                return channel.send({ type: 'MOCK_NOT_FOUND' });
            }
            // Handle a scenario when there is a request handler,
            // but it doesn't return any mocked response.
            if (!response) {
                console.warn('[MSW] Expected a mocking resolver function to return a mocked response Object, but got: %s. Original response is going to be used instead.', response);
                return channel.send({ type: 'MOCK_NOT_FOUND' });
            }
            const responseWithSerializedHeaders = Object.assign(Object.assign({}, response), { headers: lib$1.headersToList(response.headers) });
            if (!options.quiet) {
                setTimeout(() => {
                    handler.log(publicRequest, responseWithSerializedHeaders, handler, parsedRequest);
                }, response.delay);
            }
            channel.send({
                type: 'MOCK_SUCCESS',
                payload: responseWithSerializedHeaders,
            });
        }
        catch (error) {
            if (error instanceof NetworkError) {
                // Treat emulated network error differently,
                // as it is an intended exception in a request handler.
                return channel.send({
                    type: 'NETWORK_ERROR',
                    payload: {
                        name: error.name,
                        message: error.message,
                    },
                });
            }
            // Treat all the other exceptions in a request handler
            // as unintended, alerting that there is a problem needs fixing.
            channel.send({
                type: 'INTERNAL_ERROR',
                payload: {
                    status: 500,
                    body: JSON.stringify({
                        errorType: error.constructor.name,
                        message: error.message,
                        location: error.stack,
                    }),
                },
            });
        }
    });
};

function requestIntegrityCheck(context, serviceWorker) {
    return __awaiter(this, void 0, void 0, function* () {
        // Signal Service Worker to report back its integrity
        serviceWorker.postMessage('INTEGRITY_CHECK_REQUEST');
        const { payload: actualChecksum } = yield context.events.once('INTEGRITY_CHECK_RESPONSE');
        // Compare the response from the Service Worker and the
        // global variable set by webpack upon build.
        if (actualChecksum !== "65d33ca82955e1c5928aed19d1bdf3f9") {
            throw new Error(`Currently active Service Worker (${actualChecksum}) is behind the latest published one (${"65d33ca82955e1c5928aed19d1bdf3f9"}).`);
        }
        return serviceWorker;
    });
}

/**
 * Intercepts and defers any requests on the page
 * until the Service Worker instance is ready.
 * Must only be used in a browser.
 */
function deferNetworkRequestsUntil(predicatePromise) {
    // Defer any `XMLHttpRequest` requests until the Service Worker is ready.
    const originalXhrSend = window.XMLHttpRequest.prototype.send;
    window.XMLHttpRequest.prototype.send = function (...args) {
        // Keep this function synchronous to comply with `XMLHttpRequest.prototype.send`,
        // because that method is always synchronous.
        lib.until(() => predicatePromise).then(() => {
            window.XMLHttpRequest.prototype.send = originalXhrSend;
            this.send(...args);
        });
    };
    // Defer any `fetch` requests until the Service Worker is ready.
    const originalFetch = window.fetch;
    window.fetch = (...args) => __awaiter(this, void 0, void 0, function* () {
        yield lib.until(() => predicatePromise);
        window.fetch = originalFetch;
        return window.fetch(...args);
    });
}

const DEFAULT_START_OPTIONS = {
    serviceWorker: {
        url: '/mockServiceWorker.js',
        options: null,
    },
    quiet: false,
    waitUntilReady: true,
    onUnhandledRequest: 'bypass',
    findWorker: (scriptURL, mockServiceWorkerUrl) => scriptURL === mockServiceWorkerUrl,
};
const createStart = (context) => {
    /**
     * Registers and activates the mock Service Worker.
     */
    return function start(options) {
        const resolvedOptions = mergeRight(DEFAULT_START_OPTIONS, options || {});
        const startWorkerInstance = () => __awaiter(this, void 0, void 0, function* () {
            if (!('serviceWorker' in navigator)) {
                console.error(`[MSW] Failed to register a Service Worker: this browser does not support Service Workers (see https://caniuse.com/serviceworkers), or your application is running on an insecure host (consider using HTTPS for custom hostnames).`);
                return null;
            }
            // Remove all previously existing event listeners.
            // This way none of the listeners persists between Fast refresh
            // of the application's code.
            context.events.removeAllListeners();
            context.events.addListener(navigator.serviceWorker, 'message', handleRequestWith(context, resolvedOptions));
            const [, instance] = yield lib.until(() => getWorkerInstance(resolvedOptions.serviceWorker.url, resolvedOptions.serviceWorker.options, resolvedOptions.findWorker));
            if (!instance) {
                return null;
            }
            const [worker, registration] = instance;
            if (!worker) {
                if (options === null || options === void 0 ? void 0 : options.findWorker) {
                    console.error(`\
[MSW] Failed to locate the Service Worker registration using a custom "findWorker" predicate.

Please ensure that the custom predicate properly locates the Service Worker registration at "${resolvedOptions.serviceWorker.url}".
More details: https://mswjs.io/docs/api/setup-worker/start#findworker
`);
                }
                else {
                    console.error(`\
[MSW] Failed to locate the Service Worker registration.

This most likely means that the worker script URL "${resolvedOptions.serviceWorker.url}" cannot resolve against the actual public hostname (${location.host}). This may happen if your application runs behind a proxy, or has a dynamic hostname.

Please consider using a custom "serviceWorker.url" option to point to the actual worker script location, or a custom "findWorker" option to resolve the Service Worker registration manually. More details: https://mswjs.io/docs/api/setup-worker/start`);
                }
                return null;
            }
            context.worker = worker;
            context.registration = registration;
            context.events.addListener(window, 'beforeunload', () => {
                if (worker.state !== 'redundant') {
                    // Notify the Service Worker that this client has closed.
                    // Internally, it's similar to disabling the mocking, only
                    // client close event has a handler that self-terminates
                    // the Service Worker when there are no open clients.
                    worker.postMessage('CLIENT_CLOSED');
                }
                // Make sure we're always clearing the interval - there are reports that not doing this can
                // cause memory leaks in headless browser environments.
                window.clearInterval(context.keepAliveInterval);
            });
            // Check if the active Service Worker is the latest published one
            const [integrityError] = yield lib.until(() => requestIntegrityCheck(context, worker));
            if (integrityError) {
                console.error(`\
[MSW] Detected outdated Service Worker: ${integrityError.message}

The mocking is still enabled, but it's highly recommended that you update your Service Worker by running:

$ npx msw init <PUBLIC_DIR>

This is necessary to ensure that the Service Worker is in sync with the library to guarantee its stability.
If this message still persists after updating, please report an issue: https://github.com/open-draft/msw/issues\
      `);
            }
            // Signal the Service Worker to enable requests interception
            const [activationError] = yield lib.until(() => activateMocking(context, options));
            if (activationError) {
                console.error('Failed to enable mocking', activationError);
                return null;
            }
            context.keepAliveInterval = window.setInterval(() => worker.postMessage('KEEPALIVE_REQUEST'), 5000);
            return registration;
        });
        const workerRegistration = startWorkerInstance();
        // Defer any network requests until the Service Worker instance is ready.
        // This prevents a race condition between the Service Worker registration
        // and application's runtime requests (i.e. requests on mount).
        if (resolvedOptions.waitUntilReady) {
            deferNetworkRequestsUntil(workerRegistration);
        }
        return workerRegistration;
    };
};

const createStop = (context) => {
    /**
     * Signal the Service Worker to disable mocking for this client.
     * Use this an an explicit way to stop the mocking, while preserving
     * the worker-client relation. Does not affect the worker's lifecycle.
     */
    return function stop() {
        var _a;
        (_a = context.worker) === null || _a === void 0 ? void 0 : _a.postMessage('MOCK_DEACTIVATE');
        context.events.removeAllListeners();
        window.clearInterval(context.keepAliveInterval);
    };
};

function use(currentHandlers, ...handlers) {
    currentHandlers.unshift(...handlers);
}
function restoreHandlers(handlers) {
    handlers.forEach((handler) => {
        if ('shouldSkip' in handler) {
            handler.shouldSkip = false;
        }
    });
}
function resetHandlers(initialHandlers, ...nextHandlers) {
    return nextHandlers.length > 0 ? [...nextHandlers] : [...initialHandlers];
}

// Declare the list of event handlers on the module's scope
// so it persists between Fash refreshes of the application's code.
let listeners = [];
/**
 * Creates a new mock Service Worker registration
 * with the given request handlers.
 * @param {RequestHandler[]} requestHandlers List of request handlers
 * @see {@link https://mswjs.io/docs/api/setup-worker `setupWorker`}
 */
function setupWorker(...requestHandlers) {
    requestHandlers.forEach((handler) => {
        if (Array.isArray(handler))
            throw new Error(`[MSW] Failed to call "setupWorker" given an Array of request handlers (setupWorker([a, b])), expected to receive each handler individually: setupWorker(a, b).`);
    });
    const context = {
        worker: null,
        registration: null,
        requestHandlers: [...requestHandlers],
        events: {
            addListener(target, event, callback) {
                target.addEventListener(event, callback);
                listeners.push({ event, target, callback });
                return () => {
                    target.removeEventListener(event, callback);
                };
            },
            removeAllListeners() {
                for (const { target, event, callback } of listeners) {
                    target.removeEventListener(event, callback);
                }
                listeners = [];
            },
            once(type) {
                const bindings = [];
                return new Promise((resolve, reject) => {
                    const handleIncomingMessage = (event) => {
                        try {
                            const message = JSON.parse(event.data);
                            if (message.type === type) {
                                resolve(message);
                            }
                        }
                        catch (error) {
                            reject(error);
                        }
                    };
                    bindings.push(context.events.addListener(navigator.serviceWorker, 'message', handleIncomingMessage), context.events.addListener(navigator.serviceWorker, 'messageerror', reject));
                }).finally(() => {
                    bindings.forEach((unbind) => unbind());
                });
            },
        },
    };
    // Error when attempting to run this function in a NodeJS environment.
    if (isNodeProcess()) {
        throw new Error('[MSW] Failed to execute `setupWorker` in a non-browser environment. Consider using `setupServer` for NodeJS environment instead.');
    }
    return {
        start: createStart(context),
        stop: createStop(context),
        use(...handlers) {
            use(context.requestHandlers, ...handlers);
        },
        restoreHandlers() {
            restoreHandlers(context.requestHandlers);
        },
        resetHandlers(...nextHandlers) {
            context.requestHandlers = resetHandlers(requestHandlers, ...nextHandlers);
        },
        printHandlers() {
            context.requestHandlers.forEach((handler) => {
                const meta = handler.getMetaInfo();
                console.groupCollapsed(meta.header);
                console.log(`Declaration: ${meta.callFrame}`);
                console.log('Resolver: %s', handler.resolver);
                if (['rest'].includes(meta.type)) {
                    console.log('Match:', `https://mswjs.io/repl?path=${meta.mask}`);
                }
                console.groupEnd();
            });
        },
    };
}

export { compose, createResponseComposition, defaultContext, defaultResponse, response, setupWorker };
