import datetime

from crm.agency_cabinet.common.client import BaseClient
from crm.agency_cabinet.agencies.proto import common_pb2, request_pb2, agency_info_pb2, clients_pb2, analytics_pb2
from crm.agency_cabinet.agencies.common import QUEUE
from crm.agency_cabinet.agencies.common import structs
from crm.agency_cabinet.common.proto_utils import timestamp_or_none


class AgenciesClientException(Exception):
    pass


class ProtocolError(AgenciesClientException):
    pass


class AgenciesClient(BaseClient):
    queue = QUEUE

    async def ping(self) -> str:
        _, data = await self._send_message(
            request_pb2.RpcRequest(ping=common_pb2.Empty()),
            common_pb2.PingOutput
        )
        return str(data)

    async def get_agencies_info(self, agency_ids: list[int]) -> list[structs.AgencyInfo]:
        request = request_pb2.RpcRequest(
            get_agencies_info=agency_info_pb2.GetAgenciesInfo(agency_ids=agency_ids)
        )

        result, data = await self._send_message(
            request,
            agency_info_pb2.GetAgenciesInfoOutput
        )

        if result == 'result':
            return structs.GetAgenciesInfoResponse.from_proto(data).agencies

        raise ProtocolError('Unexpected response')

    async def get_all_agencies_info(self) -> list[structs.AgencyInfo]:
        request = request_pb2.RpcRequest(
            get_all_agencies_info=common_pb2.Empty()
        )

        result, data = await self._send_message(
            request,
            agency_info_pb2.GetAgenciesInfoOutput
        )

        if result == 'result':
            return structs.GetAgenciesInfoResponse.from_proto(data).agencies

        raise ProtocolError('Unexpected response')

    async def get_client_info(self, agency_id, client_id) -> structs.ClientInfo:
        request = request_pb2.RpcRequest(
            get_client_info=clients_pb2.GetClientInfo(client_id=client_id, agency_id=agency_id)
        )

        result, data = await self._send_message(
            request,
            clients_pb2.GetClientInfoOutput
        )

        if result == 'result':
            return structs.GetClientInfoResponse.from_proto(data).client
        # TODO: exception for another results
        raise ProtocolError('Unexpected response')

    async def get_clients_info(
        self,
        agency_id,
        limit: int = None,
        offset: int = None,
        search_query: str = None
    ) -> list[structs.ClientInfo]:
        request = request_pb2.RpcRequest(
            get_clients_info=structs.GetClientsInfoRequest(
                agency_id=agency_id,
                limit=limit,
                offset=offset,
                search_query=search_query
            ).to_proto()
        )

        result, data = await self._send_message(
            request,
            clients_pb2.GetClientsInfoOutput
        )

        if result == 'result':
            return structs.GetClientsInfoResponse.from_proto(data).clients

        raise ProtocolError('Unexpected response')

    async def get_average_budget_distribution(
        self,
        agency_id: int,
        date_start: datetime.date,
        date_end: datetime.date
    ) -> structs.GetAverageBudgetDistributionResponse:
        request = request_pb2.RpcRequest(
            get_average_budget_distribution=analytics_pb2.GetAverageBudgetDistribution(
                agency_id=agency_id,
                date_from=timestamp_or_none(datetime.datetime.combine(date_start.replace(day=1), datetime.time.min)),
                date_to=timestamp_or_none(datetime.datetime.combine(date_end.replace(day=1), datetime.time.min)),
            )
        )

        result, data = await self._send_message(
            request,
            analytics_pb2.GetAverageBudgetDistributionOutput
        )

        if result == 'result':
            return structs.GetAverageBudgetDistributionResponse.from_proto(data)

        raise ProtocolError('Unexpected response')

    async def get_market_situation(
        self,
        agency_id: int,
        left_date_start: datetime.date,
        left_date_end: datetime.date,
        right_date_start: datetime.date,
        right_date_end: datetime.date,
    ) -> structs.GetMarketSituationResponse:
        request = request_pb2.RpcRequest(
            get_market_situation=analytics_pb2.GetMarketSituation(
                agency_id=agency_id,
                left_date_from=timestamp_or_none(datetime.datetime.combine(left_date_start.replace(day=1), datetime.time.min)),
                left_date_to=timestamp_or_none(datetime.datetime.combine(left_date_end.replace(day=1), datetime.time.min)),
                right_date_from=timestamp_or_none(datetime.datetime.combine(right_date_start.replace(day=1), datetime.time.min)),
                right_date_to=timestamp_or_none(datetime.datetime.combine(right_date_end.replace(day=1), datetime.time.min)),
            )
        )

        result, data = await self._send_message(
            request,
            analytics_pb2.GetMarketSituationOutput
        )

        if result == 'result':
            return structs.GetMarketSituationResponse.from_proto(data)

        raise ProtocolError('Unexpected response')

    async def get_active_clients(
        self,
        agency_id: int,
        left_date_start: datetime.date,
        left_date_end: datetime.date,
        right_date_start: datetime.date,
        right_date_end: datetime.date,
    ) -> structs.GetActiveClientsResponse:
        request = request_pb2.RpcRequest(
            get_active_clients=analytics_pb2.GetActiveClients(
                agency_id=agency_id,
                left_date_from=timestamp_or_none(datetime.datetime.combine(left_date_start.replace(day=1), datetime.time.min)),
                left_date_to=timestamp_or_none(datetime.datetime.combine(left_date_end.replace(day=1), datetime.time.min)),
                right_date_from=timestamp_or_none(datetime.datetime.combine(right_date_start.replace(day=1), datetime.time.min)),
                right_date_to=timestamp_or_none(datetime.datetime.combine(right_date_end.replace(day=1), datetime.time.min)),
            )
        )

        result, data = await self._send_message(
            request,
            analytics_pb2.GetActiveClientsOutput
        )

        if result == 'result':
            return structs.GetActiveClientsResponse.from_proto(data)

        raise ProtocolError('Unexpected response')

    async def get_clients_increase(
        self,
        agency_id: int,
        left_date_start: datetime.date,
        left_date_end: datetime.date,
        right_date_start: datetime.date,
        right_date_end: datetime.date,
    ) -> structs.GetClientsIncreaseResponse:
        request = request_pb2.RpcRequest(
            get_clients_increase=analytics_pb2.GetClientsIncrease(
                agency_id=agency_id,
                left_date_from=timestamp_or_none(
                    datetime.datetime.combine(left_date_start.replace(day=1), datetime.time.min)),
                left_date_to=timestamp_or_none(
                    datetime.datetime.combine(left_date_end.replace(day=1), datetime.time.min)),
                right_date_from=timestamp_or_none(
                    datetime.datetime.combine(right_date_start.replace(day=1), datetime.time.min)),
                right_date_to=timestamp_or_none(
                    datetime.datetime.combine(right_date_end.replace(day=1), datetime.time.min)),
            )
        )

        result, data = await self._send_message(
            request,
            analytics_pb2.GetClientsIncreaseOutput
        )

        if result == 'result':
            return structs.GetClientsIncreaseResponse.from_proto(data)

        raise ProtocolError('Unexpected response')
