import datetime
from typing import Optional
from dataclasses import dataclass

from decimal import Decimal
from crm.agency_cabinet.agencies.proto import agency_info_pb2, clients_pb2, analytics_pb2
from crm.agency_cabinet.common.proto_utils import (
    BaseStruct,
    decimal_percent_to_string,
    safe_get_nullable_field
)


@dataclass
class GetAgenciesInfoRequest(BaseStruct):
    agency_ids: list[int]

    @classmethod
    def from_proto(cls, message: agency_info_pb2.GetAgenciesInfo) -> 'GetAgenciesInfoRequest':
        return cls(agency_ids=message.agency_ids)


@dataclass
class AgencyInfo(BaseStruct):
    agency_id: int
    name: str
    phone: str
    email: str
    site: str
    actual_address: str
    legal_address: str

    @classmethod
    def from_proto(cls, message: agency_info_pb2.AgencyInfo) -> 'AgencyInfo':
        return cls(
            agency_id=message.agency_id,
            name=message.name,
            phone=message.phone,
            email=message.email,
            site=message.site,
            actual_address=message.actual_address,
            legal_address=message.legal_address
        )

    def to_proto(self) -> agency_info_pb2.AgencyInfo:
        return agency_info_pb2.AgencyInfo(
            agency_id=self.agency_id,
            name=self.name,
            phone=self.phone,
            email=self.email,
            site=self.site,
            actual_address=self.actual_address,
            legal_address=self.legal_address
        )


@dataclass
class GetAgenciesInfoResponse(BaseStruct):
    agencies: list[AgencyInfo]

    @classmethod
    def from_proto(cls, message: agency_info_pb2.GetAgenciesInfoOutput) -> 'GetAgenciesInfoResponse':
        return cls(agencies=[AgencyInfo.from_proto(agency) for agency in message.agencies])

    def to_proto(self) -> agency_info_pb2.AgencyInfoList:
        return agency_info_pb2.AgencyInfoList(agencies=[agency.to_proto() for agency in self.agencies])


@dataclass
class ClientInfo(BaseStruct):
    id: int
    name: str
    login: str

    @classmethod
    def from_proto(cls, message: clients_pb2.ClientInfo) -> 'ClientInfo':
        return cls(id=message.id, name=message.name, login=message.login)

    def to_proto(self) -> clients_pb2.ClientInfo:
        return clients_pb2.ClientInfo(id=self.id, name=self.name, login=self.login)


@dataclass
class GetClientInfoResponse(BaseStruct):
    client: ClientInfo

    @classmethod
    def from_proto(cls, message: clients_pb2.ClientInfo) -> 'GetClientInfoResponse':
        return cls(client=ClientInfo.from_proto(message))

    def to_proto(self) -> clients_pb2.ClientInfo:
        return self.client.to_proto()


@dataclass
class GetClientsInfoResponse(BaseStruct):
    clients: list[ClientInfo]

    @classmethod
    def from_proto(cls, message: clients_pb2.ClientsInfoList) -> 'GetClientsInfoResponse':
        return cls(clients=[ClientInfo.from_proto(client) for client in message.clients])

    def to_proto(self) -> clients_pb2.ClientInfo:
        return clients_pb2.ClientsInfoList(clients=[client.to_proto() for client in self.clients])


@dataclass
class GetClientInfoRequest(BaseStruct):
    client_id: int
    agency_id: int

    @classmethod
    def from_proto(cls, message: clients_pb2.GetClientInfo) -> 'GetClientInfoRequest':
        return cls(agency_id=message.agency_id, client_id=message.client_id)


@dataclass
class GetClientsInfoRequest(BaseStruct):
    agency_id: int
    limit: Optional[int] = None
    offset: Optional[int] = None
    search_query: Optional[str] = None

    @classmethod
    def from_proto(cls, message: clients_pb2.GetClientsInfo) -> 'GetClientsInfoRequest':
        return cls(
            agency_id=message.agency_id,
            limit=safe_get_nullable_field(message, 'limit'),
            offset=safe_get_nullable_field(message, 'offset'),
            search_query=safe_get_nullable_field(message, 'search_query')
        )

    def to_proto(self) -> clients_pb2.GetClientsInfo:
        return clients_pb2.GetClientsInfo(
            agency_id=self.agency_id,
            limit=self.limit,
            offset=self.offset,
            search_query=self.search_query
        )


@dataclass
class GetAverageBudgetDistributionRequest(BaseStruct):
    agency_id: int
    date_from: datetime.datetime
    date_to: datetime.datetime

    @classmethod
    def from_proto(cls, message: analytics_pb2.GetAverageBudgetDistribution) -> 'GetAverageBudgetDistributionRequest':
        return cls(
            agency_id=message.agency_id,
            date_from=message.date_from.ToDatetime(),
            date_to=message.date_to.ToDatetime()
    )


@dataclass
class GetMarketSituationRequest(BaseStruct):
    agency_id: int
    left_date_from: datetime.datetime
    left_date_to: datetime.datetime

    right_date_from: datetime.datetime
    right_date_to: datetime.datetime

    @classmethod
    def from_proto(cls, message: analytics_pb2.GetMarketSituation) -> 'GetMarketSituationRequest':
        return cls(
            agency_id=message.agency_id,
            left_date_from=message.left_date_from.ToDatetime(),
            left_date_to=message.left_date_to.ToDatetime(),
            right_date_from=message.right_date_from.ToDatetime(),
            right_date_to=message.right_date_to.ToDatetime()
        )


@dataclass
class GetActiveClientsRequest(BaseStruct):
    agency_id: int
    left_date_from: datetime.datetime
    left_date_to: datetime.datetime

    right_date_from: datetime.datetime
    right_date_to: datetime.datetime

    @classmethod
    def from_proto(cls, message: analytics_pb2.GetActiveClients) -> 'GetActiveClientsRequest':
        return cls(
            agency_id=message.agency_id,
            left_date_from=message.left_date_from.ToDatetime(),
            left_date_to=message.left_date_to.ToDatetime(),
            right_date_from=message.right_date_from.ToDatetime(),
            right_date_to=message.right_date_to.ToDatetime()
        )


@dataclass
class GetClientsIncreaseRequest(BaseStruct):
    agency_id: int
    left_date_from: datetime.datetime
    left_date_to: datetime.datetime

    right_date_from: datetime.datetime
    right_date_to: datetime.datetime

    @classmethod
    def from_proto(cls, message: analytics_pb2.GetClientsIncrease) -> 'GetClientsIncreaseRequest':
        return cls(
            agency_id=message.agency_id,
            left_date_from=message.left_date_from.ToDatetime(),
            left_date_to=message.left_date_to.ToDatetime(),
            right_date_from=message.right_date_from.ToDatetime(),
            right_date_to=message.right_date_to.ToDatetime()
        )


@dataclass
class ClientsIncreasePart(BaseStruct):
    new_customers: int
    customers: int
    new_customers_prev: int
    customers_prev: int

    @classmethod
    def from_proto(cls, message: analytics_pb2.ClientsIncreasePart) -> 'ClientsIncreasePart':
        return cls(
            new_customers=message.new_customers,
            customers=message.customers,
            new_customers_prev=message.new_customers_prev,
            customers_prev=message.customers_prev
        )

    def to_proto(self) -> analytics_pb2.ClientsIncreasePart:
        return analytics_pb2.ClientsIncreasePart(
            new_customers=self.new_customers,
            customers=self.customers,
            new_customers_prev=self.new_customers_prev,
            customers_prev=self.customers_prev
        )


@dataclass
class GetClientsIncreaseResponse(BaseStruct):
    other: list[ClientsIncreasePart]
    current_at_left_date: ClientsIncreasePart
    current_at_right_date: ClientsIncreasePart
    percent_less: Decimal

    @classmethod
    def from_proto(cls, message: analytics_pb2.ClientsIncreaseInfo) -> 'GetClientsIncreaseResponse':
        return cls(
            other=[ClientsIncreasePart.from_proto(o) for o in message.other],
            current_at_left_date=ClientsIncreasePart.from_proto(message.current_at_left_date),
            current_at_right_date=ClientsIncreasePart.from_proto(message.current_at_right_date),
            percent_less=Decimal(message.percent_less)
        )

    def to_proto(self) -> analytics_pb2.ClientsIncreaseInfo:
        return analytics_pb2.ClientsIncreaseInfo(
            other=[o.to_proto() for o in self.other],
            current_at_left_date=self.current_at_left_date.to_proto(),
            current_at_right_date=self.current_at_right_date.to_proto(),
            percent_less=decimal_percent_to_string(self.percent_less)
        )


@dataclass
class ActiveClientsPart(BaseStruct):
    customers_at_left_date: int
    customers_at_right_date: int

    @classmethod
    def from_proto(cls, message: analytics_pb2.ActiveClientsPart) -> 'ActiveClientsPart':
        return cls(
            customers_at_left_date=message.customers_at_left_date,
            customers_at_right_date=message.customers_at_right_date
        )

    def to_proto(self) -> analytics_pb2.ActiveClientsPart:
        return analytics_pb2.ActiveClientsPart(
            customers_at_left_date=self.customers_at_left_date,
            customers_at_right_date=self.customers_at_right_date
        )


@dataclass
class GetActiveClientsResponse(BaseStruct):
    other: list[ActiveClientsPart]
    current: ActiveClientsPart
    percent_less: Decimal

    @classmethod
    def from_proto(cls, message: analytics_pb2.ActiveClientsInfo) -> 'GetActiveClientsResponse':
        return cls(
            other=[ActiveClientsPart.from_proto(o) for o in message.other],
            current=ActiveClientsPart.from_proto(message.current),
            percent_less=Decimal(message.percent_less)
        )

    def to_proto(self) -> analytics_pb2.ActiveClientsInfo:
        return analytics_pb2.ActiveClientsInfo(
            other=[o.to_proto() for o in self.other],
            current=self.current.to_proto(),
            percent_less=decimal_percent_to_string(self.percent_less)
        )


@dataclass
class MarketSituationPart(BaseStruct):
    average_budget: Decimal
    customers: int

    @classmethod
    def from_proto(cls, message: analytics_pb2.MarketSituationPart) -> 'MarketSituationPart':
        return cls(
            average_budget=Decimal(message.average_budget),
            customers=message.customers
        )

    def to_proto(self) -> analytics_pb2.MarketSituationPart:
        return analytics_pb2.MarketSituationPart(
            average_budget=decimal_percent_to_string(self.average_budget),
            customers=self.customers
        )


@dataclass
class GetMarketSituationResponse(BaseStruct):
    other: list[MarketSituationPart]
    current_at_left_date: MarketSituationPart
    current_at_right_date: MarketSituationPart
    percent_less: Decimal

    @classmethod
    def from_proto(cls, message: analytics_pb2.MarketSituationInfo) -> 'GetMarketSituationResponse':
        return cls(
            other=[MarketSituationPart.from_proto(part) for part in message.other],
            current_at_left_date=MarketSituationPart.from_proto(message.current_at_left_date),
            current_at_right_date=MarketSituationPart.from_proto(message.current_at_right_date),
            percent_less=Decimal(message.percent_less)
        )

    def to_proto(self) -> analytics_pb2.MarketSituationInfo:
        return analytics_pb2.MarketSituationInfo(
            other=[part.to_proto() for part in self.other],
            current_at_left_date=self.current_at_left_date.to_proto(),
            current_at_right_date=self.current_at_right_date.to_proto(),
            percent_less=decimal_percent_to_string(self.percent_less)
        )


@dataclass
class AverageBudgetMarketPiePart(BaseStruct):
    percent: Decimal
    grade: str
    customers: int

    @classmethod
    def from_proto(cls, message: analytics_pb2.MarketPiePart) -> 'AverageBudgetMarketPiePart':
        return cls(
            percent=Decimal(message.percent),
            grade=message.grade,
            customers=message.customers
        )

    def to_proto(self) -> analytics_pb2.MarketPiePart:
        return analytics_pb2.MarketPiePart(
            percent=decimal_percent_to_string(self.percent),
            grade=self.grade,
            customers=self.customers
        )


@dataclass
class GetAverageBudgetDistributionResponse(BaseStruct):
    current: list[AverageBudgetMarketPiePart]
    other: list[AverageBudgetMarketPiePart]

    median_budget_current: Decimal
    median_budget_other: Decimal

    @classmethod
    def from_proto(cls, message: analytics_pb2.AverageBudgetDistributionInfo) -> 'GetAverageBudgetDistributionResponse':
        return cls(
            current=[AverageBudgetMarketPiePart.from_proto(pie) for pie in message.current],
            other=[AverageBudgetMarketPiePart.from_proto(pie) for pie in message.other],
            median_budget_other=Decimal(message.median_budget_other),
            median_budget_current=Decimal(message.median_budget_current)
        )

    def to_proto(self) -> analytics_pb2.AverageBudgetDistributionInfo:
        return analytics_pb2.AverageBudgetDistributionInfo(
            current=[pie.to_proto() for pie in self.current],
            other=[pie.to_proto() for pie in self.other],
            median_budget_other=decimal_percent_to_string(self.median_budget_other),
            median_budget_current=decimal_percent_to_string(self.median_budget_current)
        )
