import logging
import typing

from asgiref.sync import async_to_sync
from yql.client.parameter_value_builder import YqlParameterValueBuilder

from crm.agency_cabinet.agencies.server.src import config
from crm.agency_cabinet.agencies.server.src.celery.base import celery_app as celery
from crm.agency_cabinet.agencies.server.src.db import models
from crm.agency_cabinet.common.yql.base import YqlModelLoader
from crm.agency_cabinet.common.yt.base import BaseModel
from crm.agency_cabinet.rewards.common.consts import CONTRACT_CODES_FOR_ALL

LOGGER = logging.getLogger('celery.load_agencies_info')


AGENCIES_INFO_QUERY = '''
USE hahn;

DECLARE $contract_table_name AS String;
DECLARE $client_table_name AS String;
DECLARE $person_table_name AS String;
DECLARE $contracts_codes AS List<String>;

SELECT
    v_contract_apex_full.agency_id AS agency_id,
    agency_name,
    t_person.phone AS phone,
    t_person.email AS email,
    url,
    legaladdress,
    address
FROM $contract_table_name as v_contract_apex_full
FULL JOIN $person_table_name AS t_person ON (t_person.id == v_contract_apex_full.person_id)
FULL JOIN $client_table_name AS t_client ON (t_client.agency_id == v_contract_apex_full.agency_id)
WHERE wsale_ag_prm_awrd_sc_tp IN $contracts_codes AND is_agency = 1
AND finish_dt >= '2022-03-01';
'''


class AgenciesInfoLoader(YqlModelLoader):

    def _find_duplicate(self, yt_row) -> typing.Optional[models.Agency]:
        @async_to_sync
        async def _get_agency(agency_id):
            async with self.db_bind:
                return await models.Agency.query.where(models.Agency.id == agency_id).gino.first()

        return _get_agency(self._get_column_value(yt_row, 'agency_id'))

    def _process_duplicate(self, yt_row, db_row: BaseModel):
        @async_to_sync
        async def _update_agency(agency: models.Agency, name: str, phone: str, email: str, site: str,
                                 actual_address: str, legal_address: str):
            async with self.db_bind:
                return await agency.update(
                    name=name,
                    phone=phone,
                    email=email,
                    site=site,
                    actual_address=actual_address,
                    legal_address=legal_address
                ).apply()

        _update_agency(
            db_row,
            self._get_column_value(yt_row, 'agency_name'),
            self._get_column_value(yt_row, 'phone'),
            self._get_column_value(yt_row, 'email'),
            self._get_column_value(yt_row, 'url'),
            self._get_column_value(yt_row, 'address'),
            self._get_column_value(yt_row, 'legaladdress'),
        )


@celery.task(bind=True)
def load_agencies_info_task(self):
    parameters = {
        '$contract_table_name': YqlParameterValueBuilder.make_string('//home/balance/prod/bo/v_contract_apex_full'),
        '$client_table_name': YqlParameterValueBuilder.make_string('//home/balance/prod/bo/t_client'),
        '$person_table_name': YqlParameterValueBuilder.make_string('//home/balance/prod/bo/t_person'),
        '$contracts_codes': YqlParameterValueBuilder.make_list([
            YqlParameterValueBuilder.make_string(code) for code in CONTRACT_CODES_FOR_ALL
        ]),
    }

    loader = AgenciesInfoLoader(
        table_path='//home/balance/prod/bo/v_contract_apex_full',
        model=models.Agency,
        columns_mapper={
            'id': 'agency_id',
            'name': 'agency_name',
            'phone': 'phone',
            'email': 'email',
            'site': 'url',
            'actual_address': 'address',
            'legal_address': 'legaladdress'
        },
        default_columns={},
        client_config={
            'cluster': 'hahn',
            'token': config.YT_CONFIG['TOKEN'],
            'config': {}
        },
        yql_token=config.YQL_CONFIG['TOKEN'],
        yql_query=AGENCIES_INFO_QUERY,
        yql_parameters=parameters
    )

    loader.load()
