import datetime
import typing

from dataclasses import dataclass
from decimal import Decimal
from collections import defaultdict, Counter
from crm.agency_cabinet.common.consts.analytics import AverageCheckBucket
from crm.agency_cabinet.agencies.common import structs
from crm.agency_cabinet.agencies.server.src.db.queries import (
    make_avg_check_agency_distribution_aggregation,
    make_current_vs_other_median_check_aggregation,
    make_median_check_distribution_aggregation,
    make_clients_info_aggregation,
    make_clients_info_aggregation_with_prev_period
)


@dataclass
class GetAverageBudgetDistribution:

    @staticmethod
    async def _build_distribution(agency_id, month_start, month_end, is_current):
        dist = await make_avg_check_agency_distribution_aggregation(
            agency_id=agency_id,
            month_end=month_end,
            month_start=month_start,
            count_other=not is_current
        ).gino.all()
        result = defaultdict(Counter)
        s = sum(Decimal(r.count) for r in dist)
        for r in dist:
            level = r.check_ctg
            result[level].update(
                {
                    'count': r.count,
                    'percent': Decimal(r.count) * Decimal(100) / s if s > 0 else Decimal(100)
                }
            )
        return result

    async def __call__(
        self,
        request: structs.GetAverageBudgetDistributionRequest
    ) -> structs.GetAverageBudgetDistributionResponse:
        date_from = request.date_from.date()
        date_to = request.date_to.date()
        agency_id = request.agency_id
        # TODO: check if agency exists
        result = await make_current_vs_other_median_check_aggregation(
            agency_id=agency_id,
            month_end=date_to,
            month_start=date_from
        ).gino.all()
        median_check_current = Decimal(0)
        median_check_other = Decimal(0)
        for r in result:
            if r.t == 'current':
                median_check_current = Decimal(r.median_check)
            else:
                median_check_other = Decimal(r.median_check)

        current_level_to_count = await self._build_distribution(agency_id, date_from, date_to, True)
        other_level_to_count = await self._build_distribution(agency_id, date_from, date_to, False)

        result = structs.GetAverageBudgetDistributionResponse(
            median_budget_current=median_check_current,
            median_budget_other=median_check_other,
            current=[
                structs.AverageBudgetMarketPiePart(
                    percent=Decimal(current_level_to_count[level.value]['percent']),
                    customers=current_level_to_count[level.value]['count'],
                    grade=level.value
                )
                for level in AverageCheckBucket
            ],
            other=[
                structs.AverageBudgetMarketPiePart(
                    percent=Decimal(other_level_to_count[level.value]['percent']),
                    customers=other_level_to_count[level.value]['count'],
                    grade=level.value
                )
                for level in AverageCheckBucket
            ],
        )
        return result


@dataclass
class GetMarketSituation:

    @staticmethod
    async def _build_data(
        agency_id: int,
        month_start: datetime.date,
        month_end: datetime.date
    ) -> typing.Tuple[typing.Dict[str, typing.Union[Decimal, int]], typing.List[typing.Dict[str, typing.Union[Decimal, int]]]]:
        data = await make_median_check_distribution_aggregation(month_start, month_end, agency_id).gino.all()
        other = []
        current = {'average_budget': Decimal(0), 'customers': 0}
        for r in data:
            if r.t == 'other':
                other.append({'average_budget': Decimal(r.median_check), 'customers': r.cnt_clients})
            else:
                current = {'average_budget': Decimal(r.median_check), 'customers': r.cnt_clients}
        return current, other

    async def __call__(
        self,
        request: structs.GetMarketSituationRequest
    ) -> structs.GetMarketSituationResponse:

        left_date_current, _ = await self._build_data(
            request.agency_id,
            request.left_date_from.date(),
            request.left_date_to.date()
        )
        right_date_current, other = await self._build_data(
            request.agency_id,
            request.right_date_from.date(),
            request.right_date_to.date()
        )
        count_less = 0
        other_struct = []
        for o in other:
            if o['average_budget'] < right_date_current['average_budget']:
                count_less += 1
            other_struct.append(
                structs.MarketSituationPart(
                    customers=o['customers'],
                    average_budget=o['average_budget']
                )
            )

        return structs.GetMarketSituationResponse(
            other=other_struct,
            current_at_left_date=structs.MarketSituationPart(
                customers=left_date_current['customers'],
                average_budget=left_date_current['average_budget']
            ),
            current_at_right_date=structs.MarketSituationPart(
                customers=right_date_current['customers'],
                average_budget=right_date_current['average_budget']
            ),
            percent_less=Decimal(count_less) / Decimal(len(other_struct)) * Decimal(100) if other_struct else Decimal(
                100)
        )


@dataclass
class GetActiveClients:

    @staticmethod
    async def _build_clients_info(agency_id, month_start, month_end, get_only_current=False, sort_by_new=False) -> list:
        data = await make_clients_info_aggregation(
            agency_id=agency_id,
            month_end=month_end,
            month_start=month_start,
            get_only_current=get_only_current,
            sort_by_new=sort_by_new
        ).gino.all()
        return data

    async def __call__(
        self,
        request: structs.GetActiveClientsRequest
    ) -> structs.GetActiveClientsResponse:
        left_date_data = await self._build_clients_info(
            request.agency_id,
            request.left_date_from.date(),
            request.left_date_to.date()
        )

        right_date_data = await self._build_clients_info(
            request.agency_id,
            request.right_date_from.date(),
            request.right_date_to.date()
        )

        other_map_agency_id_data = {}
        current_data = {'customers_at_left_date': 0, 'customers_at_right_date': 0}
        for row in left_date_data:
            if row.t == 'other':
                other_map_agency_id_data[row.agency_id] = {
                    'customers_at_left_date': row.cnt_clients,
                    'customers_at_right_date': 0
                }
            else:
                current_data['customers_at_left_date'] = row.cnt_clients

        count_less = 0
        less = True
        other_struct = []

        for row in right_date_data:
            if row.t == 'other':
                if less:
                    count_less += 1
                if row.agency_id in other_map_agency_id_data:
                    other_map_agency_id_data[row.agency_id]['customers_at_right_date'] = row.cnt_clients
                else:
                    other_map_agency_id_data[row.agency_id] = {
                        'customers_at_right_date': row.cnt_clients,
                        'customers_at_left_date': 0
                    }
                other_struct.append(structs.ActiveClientsPart(
                    customers_at_right_date=other_map_agency_id_data[row.agency_id]['customers_at_right_date'],
                    customers_at_left_date=other_map_agency_id_data[row.agency_id]['customers_at_left_date']
                ))
            else:
                less = False
                current_data['customers_at_right_date'] = row.cnt_clients
        return structs.GetActiveClientsResponse(
            other=other_struct,
            current=structs.ActiveClientsPart(
                customers_at_right_date=current_data['customers_at_right_date'],
                customers_at_left_date=current_data['customers_at_left_date']),
            percent_less=Decimal(count_less) /
            Decimal(
                len(other_struct)) *
            Decimal(100) if other_struct else Decimal(100))


@dataclass
class GetClientsIncrease:

    @staticmethod
    async def _build_clients_info(agency_id, month_start, month_end, get_only_current=False, sort_by_new=False) -> list:
        data = await make_clients_info_aggregation_with_prev_period(
            agency_id=agency_id,
            month_end=month_end,
            month_start=month_start,
            get_only_current=get_only_current,
            sort_by_new=sort_by_new
        ).gino.all()
        return data

    async def __call__(
        self,
        request: structs.GetClientsIncreaseRequest
    ) -> structs.GetClientsIncreaseResponse:
        # TODO: better
        left_date_from = request.left_date_from.date()
        left_date_to = request.left_date_to.date()

        right_date_to = request.right_date_to.date()
        right_date_from = request.right_date_from.date()

        left_date_data = await self._build_clients_info(
            request.agency_id,
            left_date_from,
            left_date_to,
            True
        )

        current_at_left_date = {'new_customers': 0, 'customers': 0, 'new_customers_prev': 0, 'customers_prev': 0}
        if left_date_data:
            current_at_left_date['new_customers'] = left_date_data[0].cnt_new_clients
            current_at_left_date['customers'] = left_date_data[0].cnt_clients
            current_at_left_date['new_customers_prev'] = left_date_data[0].cnt_new_clients_prev
            current_at_left_date['customers_prev'] = left_date_data[0].cnt_clients_prev

        right_date_data = await self._build_clients_info(
            request.agency_id,
            right_date_from,
            right_date_to,
        )

        other_struct = []
        current_struct = structs.ClientsIncreasePart(
            new_customers=0,
            customers=0,
            new_customers_prev=0,
            customers_prev=0
        )

        for row in right_date_data:
            if row.t == 'other':
                other_struct.append(
                    structs.ClientsIncreasePart(
                        new_customers=row.cnt_new_clients,
                        customers=row.cnt_clients,
                        new_customers_prev=row.cnt_new_clients_prev,
                        customers_prev=row.cnt_clients_prev
                    )
                )
            else:
                current_struct = structs.ClientsIncreasePart(
                    new_customers=row.cnt_new_clients,
                    customers=row.cnt_clients,
                    new_customers_prev=row.cnt_new_clients_prev,
                    customers_prev=row.cnt_clients_prev
                )
        current_increase = current_struct.new_customers - current_struct.new_customers_prev
        count_less = sum(1 for struct in other_struct if current_increase > struct.new_customers - struct.new_customers_prev)
        return structs.GetClientsIncreaseResponse(
            other=other_struct,
            current_at_left_date=structs.ClientsIncreasePart(
                new_customers=current_at_left_date['new_customers'],
                customers=current_at_left_date['customers'],
                new_customers_prev=current_at_left_date['new_customers_prev'],
                customers_prev=current_at_left_date['customers_prev']
            ),
            current_at_right_date=current_struct,
            percent_less=Decimal(count_less) / Decimal(len(other_struct)) * Decimal(100) if other_struct else Decimal(100))
