from typing import Optional

from crm.agency_cabinet.certificates.client.lib.exceptions import CertificateNotFound
from crm.agency_cabinet.certificates.common import QUEUE
from crm.agency_cabinet.certificates.common.structs import (
    AgencyCertificate,
    AgencyCertificateDetails,
    AgencyCertificateDetailsRequest,
    AgencyCertificatesHistoryEntry,
    CertifiedEmployee,
    CertifiedEmployees,
)
from crm.agency_cabinet.certificates.proto import (
    certificates_pb2,
    common_pb2,
    request_pb2,
)
from crm.agency_cabinet.common.client import BaseClient
from crm.agency_cabinet.common.client.exceptions import ProtocolError


class Client(BaseClient):
    queue = QUEUE

    async def ping(self) -> str:
        _, data = await self._send_message(
            request_pb2.RpcRequest(ping=common_pb2.Empty()),
            common_pb2.PingOutput,
        )
        return str(data)

    async def list_agency_certificates(self, agency_id: int) -> list[AgencyCertificate]:
        request = request_pb2.RpcRequest(
            list_agency_certificates=(
                certificates_pb2.ListAgencyCertificatesRequest(agency_id=agency_id)
            )
        )

        result, data = await self._send_message(
            request, certificates_pb2.ListAgencyCertificatesResponse
        )

        if result == "certificates":
            return [AgencyCertificate.from_proto(cert) for cert in data.certificates]

        raise ProtocolError(f"Unknown response {result}")

    async def fetch_agency_certificates_history(
        self,
        agency_id: int,
        project: Optional[str],
        offset: int,
        limit: int,
    ) -> list[AgencyCertificatesHistoryEntry]:
        request = request_pb2.RpcRequest(
            fetch_agency_certificates_history=(
                certificates_pb2.FetchAgencyCertificatesHistoryRequest(
                    agency_id=agency_id,
                    project=project,
                    pagination=common_pb2.Pagination(offset=offset, limit=limit),
                )
            )
        )

        result, data = await self._send_message(
            request,
            certificates_pb2.FetchAgencyCertificatesHistoryResponse,
        )

        if result == "certificates":
            return [
                AgencyCertificatesHistoryEntry.from_proto(cert)
                for cert in data.certificates
            ]

        raise ProtocolError(f"Unknown response {result}")

    async def list_employees_certificates(
        self,
        agency_id: int,
        offset: int,
        limit: int,
        project: Optional[str],
        search_query: Optional[str],
        status: Optional[str],
    ) -> list[CertifiedEmployee]:
        request = request_pb2.RpcRequest(
            list_employees_certificates=certificates_pb2.ListEmployeesCertificatesRequest(  # noqa
                agency_id=agency_id,
                pagination=common_pb2.Pagination(offset=offset, limit=limit),
                project=project,
                search_query=search_query,
                status=status,
            )
        )

        result, data = await self._send_message(
            request, certificates_pb2.ListEmployeesCertificatesResponse
        )

        if result == "employees":
            return CertifiedEmployees.from_proto(data).employees

        raise ProtocolError(f"Unknown response {result}")

    async def fetch_agency_certificate_details(
        self, agency_id: int
    ) -> AgencyCertificateDetails:
        message = request_pb2.RpcRequest(
            fetch_agency_certificate_details=AgencyCertificateDetailsRequest(
                agency_id=agency_id
            ).to_proto()
        )

        result, data = await self._send_message(
            message, certificates_pb2.AgencyCertificateDetailsResponse
        )

        if result == "certificate_not_found":
            raise CertificateNotFound(agency_id=agency_id)

        if result == "details":
            return AgencyCertificateDetails.from_proto(data)

        raise ProtocolError(f"Unknown response {result}")
