from dataclasses import dataclass
from datetime import datetime
from crm.agency_cabinet.common.enum import BaseEnum
from typing import List, Optional

from crm.agency_cabinet.certificates.common.structs.direct_details import (
    DirectBonusPoint,
    DirectCertificationCondition,
    DirectCertificationScores,
    DirectKPI,
)
from crm.agency_cabinet.certificates.proto import certificates_pb2
from smb.common.helpers import Converter, PbDatetimeConverter

__all__ = [
    "EmployeeCertificate",
    "AgencyCertificateDetails",
    "AgencyCertificate",
    "CertifiedEmployees",
    "CertifiedEmployee",
    "AgencyCertificatesHistoryEntry",
    "EmployeeCertificateStatus",
]


class EmployeeCertificateStatus(BaseEnum):
    EXPIRED = "expired"
    EXPIRES_IN_SEMIYEAR = "expires_in_semiyear"
    ACTIVE = "active"


@dataclass
class AgencyCertificate:
    id: int
    project: str
    expiration_time: datetime
    auto_renewal_is_met: bool

    @classmethod
    def from_proto(
        cls, data: certificates_pb2.AgencyCertificate
    ) -> "AgencyCertificate":
        return cls(
            id=data.id,
            project=data.project,
            expiration_time=PbDatetimeConverter().to_datetime(data.expiration_time),
            auto_renewal_is_met=data.auto_renewal_is_met,
        )

    def to_proto(self) -> certificates_pb2.AgencyCertificate:
        message = certificates_pb2.AgencyCertificate(
            id=self.id,
            project=self.project,
            expiration_time=PbDatetimeConverter().from_datetime(self.expiration_time),
            auto_renewal_is_met=self.auto_renewal_is_met,
        )

        return message


@dataclass
class AgencyCertificatesHistoryEntry:
    id: int
    project: str
    start_time: datetime
    expiration_time: datetime

    @classmethod
    def from_proto(
        cls, data: certificates_pb2.AgencyCertificatesHistoryEntry
    ) -> "AgencyCertificatesHistoryEntry":
        return cls(
            id=data.id,
            project=data.project,
            start_time=PbDatetimeConverter().to_datetime(data.start_time),
            expiration_time=PbDatetimeConverter().to_datetime(data.expiration_time),
        )

    def to_proto(self) -> certificates_pb2.AgencyCertificatesHistoryEntry:
        return certificates_pb2.AgencyCertificatesHistoryEntry(
            id=self.id,
            project=self.project,
            start_time=PbDatetimeConverter().from_datetime(self.start_time),
            expiration_time=PbDatetimeConverter().from_datetime(self.expiration_time),
        )


employee_certificate_status_converter = Converter(
    [
        (
            EmployeeCertificateStatus.EXPIRED,
            certificates_pb2.EmployeeCertificate.StatusType.EXPIRED,
        ),
        (
            EmployeeCertificateStatus.EXPIRES_IN_SEMIYEAR,
            certificates_pb2.EmployeeCertificate.StatusType.EXPIRES_IN_SEMIYEAR,
        ),
        (
            EmployeeCertificateStatus.ACTIVE,
            certificates_pb2.EmployeeCertificate.StatusType.ACTIVE,
        ),
    ]
)


@dataclass
class EmployeeCertificate:
    project: str
    expiration_time: datetime
    start_time: datetime
    status: EmployeeCertificateStatus
    external_id: str

    @classmethod
    def from_proto(
        cls, message: certificates_pb2.EmployeeCertificate
    ) -> "EmployeeCertificate":
        return cls(
            project=message.project,
            expiration_time=PbDatetimeConverter.to_datetime(message.expiration_time),
            start_time=PbDatetimeConverter.to_datetime(message.start_time),
            external_id=message.external_id,
            status=employee_certificate_status_converter.reversed(message.status),
        )

    def to_proto(self) -> certificates_pb2.EmployeeCertificate:
        return certificates_pb2.EmployeeCertificate(
            project=self.project,
            start_time=PbDatetimeConverter.from_datetime(self.start_time),
            expiration_time=PbDatetimeConverter.from_datetime(self.expiration_time),
            external_id=self.external_id,
            status=employee_certificate_status_converter.forward(self.status),
        )


@dataclass
class CertifiedEmployee:
    email: str
    agency_id: int
    certificates: list[EmployeeCertificate]
    name: Optional[str] = None

    @classmethod
    def from_proto(
        cls, message: certificates_pb2.CertifiedEmployee
    ) -> "CertifiedEmployee":
        return cls(
            name=message.name if message.HasField("name") else None,
            email=message.email,
            agency_id=message.agency_id,
            certificates=[
                EmployeeCertificate.from_proto(cert) for cert in message.certificates
            ],
        )

    def to_proto(self) -> certificates_pb2.CertifiedEmployee:
        return certificates_pb2.CertifiedEmployee(
            name=self.name,
            email=self.email,
            agency_id=self.agency_id,
            certificates=[cert.to_proto() for cert in self.certificates],
        )


@dataclass
class CertifiedEmployees:
    employees: list[CertifiedEmployee]

    @classmethod
    def from_proto(
        cls, message: certificates_pb2.CertifiedEmployees
    ) -> "CertifiedEmployees":
        return cls(
            employees=[CertifiedEmployee.from_proto(ec) for ec in message.employees]
        )

    def to_proto(self) -> certificates_pb2.CertifiedEmployees:
        return certificates_pb2.CertifiedEmployees(
            employees=[ec.to_proto() for ec in self.employees]
        )


@dataclass
class AgencyCertificateDetails:
    agency_id: int
    conditions: List[DirectCertificationCondition]
    kpis: List[DirectKPI]
    bonus_points: List[DirectBonusPoint]
    scores: List[DirectCertificationScores]

    @classmethod
    def from_proto(cls, data: certificates_pb2.AgencyCertificateDetails):
        return cls(
            agency_id=data.agency_id,
            conditions=[
                DirectCertificationCondition.from_proto(cond)
                for cond in data.conditions
            ],
            kpis=[DirectKPI.from_proto(kpi) for kpi in data.kpis],
            bonus_points=[
                DirectBonusPoint.from_proto(bonus_point)
                for bonus_point in data.bonus_points
            ],
            scores=[
                DirectCertificationScores.from_proto(score) for score in data.scores
            ],
        )

    def to_proto(self) -> certificates_pb2.AgencyCertificateDetails:
        return certificates_pb2.AgencyCertificateDetails(
            agency_id=self.agency_id,
            conditions=[cond.to_proto() for cond in self.conditions],
            kpis=[kpi.to_proto() for kpi in self.kpis],
            bonus_points=[bonus_point.to_proto() for bonus_point in self.bonus_points],
            scores=[score.to_proto() for score in self.scores],
        )
