import asyncio

from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient
from yt.wrapper.default_config import retries_config

from crm.agency_cabinet.certificates.server.lib.celery import celery_app
from crm.agency_cabinet.certificates.server.lib.config import (
    CERTIFICATES_LOADER_SETTINGS as LOADER_CONFIG,
)
from crm.agency_cabinet.certificates.server.lib.db.engine import DB
from crm.agency_cabinet.common.server.rpc.config import DsnServerConfig

from .load_certificates_data import (
    AgencyDirectCertificatesDetailsLoader,
    EmployeeCertificatesDataLoader,
)

__all__ = [
    "load_employee_certificates_data_task",
    "load_direct_details_certificates_data_task",
]


@celery_app.task(bind=True)
def load_employee_certificates_data_task(*args, **kwargs):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_load_employee_certificates_data(cfg))


@celery_app.task(bind=True)
def load_direct_details_certificates_data_task(*args, **kwargs):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_load_direct_details_certificate_data(cfg))


async def _load_employee_certificates_data(cfg: DsnServerConfig):
    db = await DB.create(cfg.get_dsn())
    yql_client = YqlClient(token=LOADER_CONFIG["yql_token"])

    await EmployeeCertificatesDataLoader(db, yql_client)()


async def _load_direct_details_certificate_data(cfg: DsnServerConfig):
    db = await DB.create(cfg.get_dsn())
    yql_client = YqlClient(token=LOADER_CONFIG["yql_token"])
    yt_client = YtClient(
        proxy=LOADER_CONFIG["yt_proxy"],
        token=LOADER_CONFIG["yt_token"],
        config={
            "proxy": {
                "retries": retries_config(
                    count=6,
                    enable=True,
                    total_timeout=None,
                    backoff={
                        "policy": "exponential",
                        "exponential_policy": {
                            "start_timeout": 30000,  # ms
                            "base": 2,
                            "max_timeout": LOADER_CONFIG["yt_timeout"] * 1000,  # ms
                            "decay_factor_bound": 0.3,
                        },
                    },
                ),
                "connect_timeout": LOADER_CONFIG["yt_timeout"] * 1000,  # ms
                "request_timeout": LOADER_CONFIG["yt_timeout"] * 1000,  # ms
                "heavy_request_timeout": LOADER_CONFIG["yt_timeout"] * 1000,  # ms
            }
        },
    )

    await AgencyDirectCertificatesDetailsLoader(
        db=db, yql_client=yql_client, yt_client=yt_client
    )()
