from typing import Type

from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder
from yt.wrapper import YtClient

from crm.agency_cabinet.certificates.server.lib.celery.tasks.load_certificates_data.synchronizers import (
    DirectBonusPointsSynchronizer,
    DirectConditionsSynchronizer,
    DirectKPISynchronizer,
    DirectProlongationScoreSynchronizer,
    EmployeesCertificatesSynchronizer,
    ProcessingSynchronizer,
)
from smb.common.pgswim import PoolType, SwimEngine

from .yql.agency_certificate_details import YQL


class EmployeeCertificatesDataLoader:
    employees_certificates_table = "//home/search-research/yateika/help/cert"
    _yql = r"""
        USE hahn;

        DECLARE $employees_certificates_table AS String;

        $employees_certificates = (
            SELECT *
            FROM (
                SELECT certId as external_id,
                    userLogin as employee_email,
                    name as employee_name,
                    agency_id,
                    COALESCE(
                        IF(is_direct = TRUE, 'direct'),
                        IF(is_metrika = TRUE, 'metrika'),
                        IF(is_media = TRUE, 'media')
                    ) AS project,
                    confirmedDate as start_time,
                    dueDate as expiration_time
                FROM $employees_certificates_table
            )
            WHERE project IS NOT NULL
        );

        SELECT external_id, employee_email, employee_name, agency_id, project, start_time, expiration_time
        FROM $employees_certificates
        ORDER BY employee_name;
    """

    def __init__(self, db: SwimEngine, yql_client: YqlClient):
        self._db = db
        self._yql_client = yql_client

    async def __call__(self) -> None:
        request = self._yql_client.query(self._yql)

        parameters = {
            "$employees_certificates_table": YqlParameterValueBuilder.make_string(
                self.employees_certificates_table
            ),
        }
        request.run(parameters=YqlParameterValueBuilder.build_json_map(parameters))
        results = request.get_results()

        if results.status == "ERROR":
            raise Exception(f"YQL failed: {results.text}")

        employees_certificates = results.table.get_iterator()

        async with self._db.acquire(PoolType.master) as con:
            async with con.transaction():
                await EmployeesCertificatesSynchronizer(con).process_data(
                    employees_certificates
                )


class AgencyDirectCertificatesDetailsLoader:
    def __init__(self, db: SwimEngine, yql_client: YqlClient, yt_client: YtClient):
        self._db = db
        self._yql_client = yql_client
        self._yt_client = yt_client

    async def _process_data(
        self, yt_table, synchronizer_class: Type[ProcessingSynchronizer]
    ):
        read_table_iter = self._yt_client.read_table(table=yt_table)
        async with self._db.acquire(PoolType.master) as con:
            async with con.transaction():
                await synchronizer_class(con).process_data(read_table_iter)

    async def __call__(self) -> None:
        request = self._yql_client.query(YQL)
        request.run()
        results = request.get_results()

        if results.status == "ERROR":
            raise Exception(f"YQL failed: {results.text}")

        yt_table = f"//{results.table.refs[0][1]}"
        for synchronizer in {
            DirectBonusPointsSynchronizer,
            DirectConditionsSynchronizer,
            DirectKPISynchronizer,
            DirectProlongationScoreSynchronizer,
        }:
            await self._process_data(yt_table=yt_table, synchronizer_class=synchronizer)
