from abc import ABC, abstractmethod
from datetime import timezone
from typing import Generator

from dateutil import parser

__all__ = [
    "AgencyCertificateProcessor",
    "DirectBonusPointProcessor",
    "DirectConditionProcessor",
    "DirectKPIProcessor",
    "Processor",
    "DirectProlongationScoreProcessor",
]


class Processor(ABC):
    @classmethod
    @abstractmethod
    def process_row(cls, row: dict) -> Generator:
        pass


class DirectKPIProcessor(Processor):
    """Обработка столбцов, содежращих данные про выполнение агентством KPI.
    За один раз обрабатывается одна строка исходной таблицы.
    Названия столбцов, которые нужно упаковать и записать в нашу таблицу про DirectKPI берутся из `meta_data`
    Затем в оригинальной таблице ищутся столбцы с таким названием, дополняются инфой из метаданных и отправляются
        на синхронизацию.
    """

    @staticmethod
    def _process_row(
        prefix: str, row: dict, meta_data: dict
    ) -> tuple[int, str, float, float, str]:
        value = float(row[prefix])
        max_value = float(meta_data[prefix]["max_score"])
        group = meta_data[prefix]["value_type"]

        return row["agency_id"], prefix, value, max_value, group

    @staticmethod
    def _parse_meta(meta_data):
        parsed_meta = {}
        for meta_row in meta_data:
            meta_row_dict = {
                row_label: row_value[0] for row_label, row_value in meta_row
            }
            parsed_meta[meta_row_dict["label"]] = meta_row_dict

        return parsed_meta

    @classmethod
    def process_row(cls, row: dict) -> Generator:
        meta_data = cls._parse_meta(row["meta_data"])

        for kpi_prefix in meta_data.keys():
            try:
                kpi = cls._process_row(kpi_prefix, row, meta_data)
                yield kpi
            except KeyError:
                pass


class DirectConditionProcessor(Processor):
    """Обработка столбцов, содежращих данные про выполнение агентством условий участия в сертификации.
    В свойстве `prefixes` хранится знание о том, какие префиксы у нужных нам столбцов.
    За один раз обрабатывается одна строка исходной таблицы.
    Во время обработки строка из исходной таблицы процессор пытается получить нужную пару столбцов
        (prefix_value - prefix_threshold) и упаковать ее в нашу таблицу DirectConditions,
        дополнив айди агентства и информацией о том, выполнено ли условие
    """

    prefixes = [
        "agency_active_clients",
        "agency_metrika_sert_spec",
        "agency_direct_sert_spec",
    ]

    @staticmethod
    def _process_row(prefix: str, row: dict) -> tuple[int, str, str, str, bool]:
        value = row[f"{prefix}_value"]
        threshold = row[f"{prefix}_threshold"]

        is_met = float(value) >= float(threshold)

        return row["agency_id"], prefix, str(value), str(threshold), is_met

    @classmethod
    def process_row(cls, row: dict) -> Generator:
        for condition_prefix in cls.prefixes:
            try:
                condition = cls._process_row(condition_prefix, row)
                yield condition
            except KeyError:
                pass


class DirectBonusPointProcessor(Processor):
    """Обработка столбцов, содежращих данные про получение агентством бонусных баллов.
    В свойстве `prefixes` хранится знание о том, какие префиксы у нужных нам столбцов.
    За один раз обрабатывается одна строка исходной таблицы.
    Во время обработки строка из исходной таблицы процессор пытается получить нужную тройку столбцов
        (prefix_value - prefix_threshold - prefix_score) и упаковать ее в нашу таблицу DirectBonuses,
        дополнив айди агентства и информацией о том, выполнено ли условие получения балла
    """

    prefixes = [
        "agency_case",
        "media_general",
        "media_revenue_bonus",
        "agency_media_sert",
        "pdz",
    ]

    @staticmethod
    def _process_row(prefix: str, row: dict) -> tuple[int, str, str, str, float, bool]:
        value = row[f"{prefix}_value"]
        threshold = row[f"{prefix}_threshold"]
        score = float(row[f"{prefix}_score"])

        try:
            is_met = float(value) >= float(threshold)
        except ValueError:
            is_met = score > 0

        return row["agency_id"], prefix, str(value), str(threshold), score, is_met

    @classmethod
    def process_row(cls, row: dict) -> Generator:
        for bonus_prefix in cls.prefixes:
            try:
                bonus_point = cls._process_row(bonus_prefix, row)
                yield bonus_point
            except KeyError:
                pass


class DirectProlongationScoreProcessor(Processor):
    prefixes = ["general", "rsya", "search"]

    @classmethod
    def process_row(cls, row: dict) -> Generator:
        for score_prefix in cls.prefixes:
            yield row["agency_id"], "direct", row[f"{score_prefix}_curr"], row[
                f"{score_prefix}_min"
            ], score_prefix


class AgencyCertificateProcessor(Processor):
    @classmethod
    def process_row(cls, row: dict) -> Generator:
        yield (
            row["agency_id"],
            row["external_id"],
            row["project"],
            parser.isoparse(row["confirmed_date"]).astimezone(timezone.utc),
            parser.isoparse(row["due_date"]).astimezone(timezone.utc),
        )
