from abc import ABC, abstractmethod
from datetime import timezone
from typing import Union

from asyncpg import Connection
from dateutil import parser

from crm.agency_cabinet.certificates.server.lib.celery.tasks.load_certificates_data.processors import (
    AgencyCertificateProcessor,
    DirectBonusPointProcessor,
    DirectConditionProcessor,
    DirectKPIProcessor,
    DirectProlongationScoreProcessor,
    Processor,
)

__all__ = [
    "AgencyCertificatesSynchronizer",
    "DirectBonusPointsSynchronizer",
    "DirectConditionsSynchronizer",
    "DirectKPISynchronizer",
    "ProcessingSynchronizer",
    "DirectProlongationScoreSynchronizer",
    "EmployeesCertificatesSynchronizer",
]


class TableSynchronizer(ABC):
    DB_TABLE: str
    COLUMNS: list[str]
    WRITE_CHUNK_SIZE: int = 100_000

    def __init__(self, con: Connection):
        self._con = con

    async def process_data(self, rows: list[Union[tuple, dict]]) -> None:
        await self.clean_existing_data()
        await self.create_tmp_table()
        rows_to_add_chunk = []
        for row in rows:
            rows_to_add = self.process_row(row)

            rows_to_add_chunk.extend(rows_to_add)
            if len(rows_to_add_chunk) == self.WRITE_CHUNK_SIZE:
                await self._write_rows(rows_to_add_chunk)
                rows_to_add_chunk.clear()

        if rows_to_add_chunk:
            await self._write_rows(rows_to_add_chunk)

        await self.write_to_table()

    async def _write_rows(self, records: list[tuple]) -> None:
        await self._con.copy_records_to_table(
            self._tmp_db_table,
            records=records,
            columns=self.COLUMNS,
        )

    async def clean_existing_data(self) -> None:
        pass

    async def create_tmp_table(self) -> None:
        await self._con.execute(
            f"""
            CREATE TEMPORARY TABLE {self._tmp_db_table} ON COMMIT DROP AS (
                SELECT {self._columns_list}
                FROM {self.DB_TABLE}
                WHERE False
            );
            """
        )

    async def write_to_table(self):
        await self._con.execute(
            f"""
            INSERT INTO {self.DB_TABLE} ({self._columns_list})
            SELECT {self._columns_list}
            FROM {self._tmp_db_table}
            """
        )

    @abstractmethod
    def process_row(self, row: Union[tuple, dict]) -> list[tuple]:
        ...

    @property
    def _tmp_db_table(self) -> str:
        return f"{self.DB_TABLE}_tmp"

    @property
    def _columns_list(self) -> str:
        return ", ".join(self.COLUMNS)


class EmployeesCertificatesSynchronizer(TableSynchronizer):
    DB_TABLE = "employee_certificates"
    COLUMNS = [
        "external_id",
        "employee_email",
        "employee_name",
        "agency_id",
        "project",
        "start_time",
        "expiration_time",
    ]

    async def clean_existing_data(self) -> None:
        await self._con.execute(
            f"""
            DELETE FROM {self.DB_TABLE}
            """
        )

    def process_row(self, row: Union[tuple, dict]) -> list[tuple]:
        return [
            tuple(
                map(
                    lambda v: str(v, 'utf-8') if isinstance(v, bytes) else v,
                    (
                        row[0],
                        row[1],
                        row[2],
                        row[3],
                        row[4],
                        parser.isoparse(row[5]).astimezone(timezone.utc),
                        parser.isoparse(row[6]).astimezone(timezone.utc),
                    )
                )
            )
        ]


class ProcessingSynchronizer(TableSynchronizer):
    PROCESSOR: Processor

    async def clean_existing_data(self) -> None:
        await self._con.execute(
            f"""
            DELETE FROM {self.DB_TABLE}
            """
        )

    def process_row(self, row: Union[tuple, dict]) -> list[tuple]:
        return list(self.PROCESSOR.process_row(row))


class DirectKPISynchronizer(ProcessingSynchronizer):
    DB_TABLE = "agency_certificates_direct_kpi"
    COLUMNS = [
        "agency_id",
        "name",
        "value",
        "max_value",
        "group_name",
    ]
    PROCESSOR = DirectKPIProcessor


class DirectBonusPointsSynchronizer(ProcessingSynchronizer):
    DB_TABLE = "agency_certificates_direct_bonus_scores"
    COLUMNS = [
        "agency_id",
        "name",
        "value",
        "threshold",
        "score",
        "is_met",
    ]
    PROCESSOR = DirectBonusPointProcessor


class DirectConditionsSynchronizer(ProcessingSynchronizer):
    DB_TABLE = "agency_certificates_direct_conditions"
    COLUMNS = [
        "agency_id",
        "name",
        "value",
        "threshold",
        "is_met",
    ]
    PROCESSOR = DirectConditionProcessor


class DirectProlongationScoreSynchronizer(ProcessingSynchronizer):
    DB_TABLE = "agency_certificates_prolongation_score"
    COLUMNS = [
        "agency_id",
        "project",
        "current_score",
        "target_score",
        "score_group",
    ]
    PROCESSOR = DirectProlongationScoreProcessor


class AgencyCertificatesSynchronizer(ProcessingSynchronizer):
    DB_TABLE = "agency_certificates"
    COLUMNS = [
        "agency_id",
        "external_id",
        "project",
        "start_time",
        "expiration_time",
    ]
    PROCESSOR = AgencyCertificateProcessor
