from crm.agency_cabinet.certificates.common.structs import (
    AgencyCertificateDetailsRequest,
    FetchAgencyCertificatesHistoryRequest,
    ListEmployeesCertificatesRequest,
)
from crm.agency_cabinet.certificates.proto import (
    certificates_pb2,
    common_pb2,
    errors_pb2,
    request_pb2,
)
from crm.agency_cabinet.certificates.server.lib.db.engine import DB
from crm.agency_cabinet.certificates.server.lib.exceptions import (
    AgencyCertificateNotFound,
)
from smb.common.rmq.rpc.server import BaseRpcHandler

from .procedures import (
    FetchAgencyCertificateDetails,
    FetchAgencyCertificatesHistory,
    ListAgencyCertificates,
    ListEmployeesCertificates,
)


class Handler(BaseRpcHandler):
    _request_proto = request_pb2.RpcRequest

    def __init__(self, db: DB):
        self._list_actual_certificates_proc = ListAgencyCertificates(db=db)
        self._fetch_agency_certificates_history_proc = FetchAgencyCertificatesHistory(
            db=db
        )
        self._list_employees_certificates_proc = ListEmployeesCertificates(db=db)
        self._fetch_agency_certificate_details = FetchAgencyCertificateDetails(db=db)

    # noinspection PyMethodMayBeStatic
    async def ping(self, _: common_pb2.Empty) -> common_pb2.PingOutput:
        return common_pb2.PingOutput(ping="pong")

    async def list_agency_certificates(
        self, message: certificates_pb2.ListEmployeesCertificatesRequest
    ) -> certificates_pb2.ListAgencyCertificatesResponse:
        certificates = await self._list_actual_certificates_proc(
            agency_id=message.agency_id
        )

        return certificates_pb2.ListAgencyCertificatesResponse(
            certificates=certificates_pb2.AgencyCertificates(
                certificates=[cert.to_proto() for cert in certificates]
            )
        )

    async def fetch_agency_certificates_history(
        self, message: certificates_pb2.FetchAgencyCertificatesHistoryRequest
    ) -> certificates_pb2.FetchAgencyCertificatesHistoryResponse:
        request = FetchAgencyCertificatesHistoryRequest.from_proto(message)
        certificates = await self._fetch_agency_certificates_history_proc(
            request=request
        )

        return certificates_pb2.FetchAgencyCertificatesHistoryResponse(
            certificates=certificates_pb2.AgencyCertificatesHistory(
                certificates=[cert.to_proto() for cert in certificates]
            )
        )

    async def list_employees_certificates(
        self, message: certificates_pb2.ListEmployeesCertificatesRequest
    ) -> certificates_pb2.ListEmployeesCertificatesResponse:
        request = ListEmployeesCertificatesRequest.from_proto(message)
        certificates = await self._list_employees_certificates_proc(
            request=request
        )

        return certificates_pb2.ListEmployeesCertificatesResponse(
            employees=certificates.to_proto()
        )

    async def fetch_agency_certificate_details(
        self, message: certificates_pb2.AgencyCertificateDetailsRequest
    ) -> certificates_pb2.AgencyCertificateDetailsResponse:
        request = AgencyCertificateDetailsRequest.from_proto(message)

        try:
            details = await self._fetch_agency_certificate_details(request=request)

            return certificates_pb2.AgencyCertificateDetailsResponse(
                details=details.to_proto()
            )
        except AgencyCertificateNotFound as not_found_err:
            return certificates_pb2.AgencyCertificateDetailsResponse(
                certificate_not_found=errors_pb2.CertificateNotFound(
                    agency_id=not_found_err.agency_id,
                )
            )
