from datetime import datetime
from typing import Optional

from crm.agency_cabinet.client_bonuses.common import QUEUE, structs
from crm.agency_cabinet.client_bonuses.proto import bonuses_pb2, common_pb2, request_pb2, reports_pb2
from crm.agency_cabinet.common.client import BaseClient
from crm.agency_cabinet.common.client.exceptions import ProtocolError

from .exceptions import (
    ClientNotFound, FileNotFoundException, UnsuitableAgencyException,
    ReportNotReadyException, NoSuchReportException
)

__all__ = ["Client"]


class Client(BaseClient):
    queue = QUEUE

    async def ping(self) -> str:
        _, data = await self._send_message(
            request_pb2.RpcRequest(ping=common_pb2.Empty()),
            common_pb2.PingOutput,
        )
        return str(data)

    async def fetch_bonuses_details(
        self,
        client_id: int,
        agency_id: int,
        datetime_start: datetime,
        datetime_end: datetime,
    ) -> list[structs.BonusDetails]:
        request = request_pb2.RpcRequest(
            fetch_bonuses_details=structs.FetchBonusesDetailsInput(
                client_id=client_id,
                agency_id=agency_id,
                datetime_start=datetime_start,
                datetime_end=datetime_end,
            ).to_proto()
        )

        result, data = await self._send_message(
            request, bonuses_pb2.FetchBonusesDetailsOutput
        )

        if result == "details":
            return structs.BonusDetailsList.from_proto(data).items
        elif result == "client_not_found":
            raise ClientNotFound(agency_id=data.agency_id, client_id=data.client_id)

        raise ProtocolError("Unexpected response")

    async def list_clients_bonuses(
        self,
        agency_id: int,
        client_type: structs.ClientType,
        bonus_type: structs.BonusType,
        limit: int,
        offset: int,
        datetime_start: datetime,
        datetime_end: datetime,
        search_query: Optional[str] = None,
    ) -> list[structs.ClientBonus]:

        request = request_pb2.RpcRequest(
            list_clients_bonuses=structs.ListClientsBonusesInput(
                agency_id=agency_id,
                client_type=client_type,
                bonus_type=bonus_type,
                limit=limit,
                offset=offset,
                datetime_start=datetime_start,
                datetime_end=datetime_end,
                search_query=search_query,
            ).to_proto()
        )

        result, data = await self._send_message(
            request, bonuses_pb2.ListClientsBonusesOutput
        )

        if result == "bonuses":
            return [structs.ClientBonus.from_proto(bonus) for bonus in data.bonuses]

        raise ProtocolError("Unexpected response")

    async def get_clients_bonuses_settings(self, agency_id: int) -> structs.ClientBonusSettings:

        request = request_pb2.RpcRequest(
            get_clients_bonuses_settings=structs.GetClientsBonusesSettingsInput(
                agency_id=agency_id,
            ).to_proto()
        )

        result, data = await self._send_message(
            request, bonuses_pb2.GetClientsBonusesSettingsOutput
        )

        if result == "settings":
            return structs.ClientBonusSettings.from_proto(data)

        raise ProtocolError("Unexpected response")

    async def fetch_client_graph(
        self,
        client_id: int,
        agency_id: int,
    ) -> structs.ClientGraph:
        request = request_pb2.RpcRequest(
            fetch_client_bonuses_graph=bonuses_pb2.FetchClientBonusesGraphInput(
                client_id=client_id,
                agency_id=agency_id,
            )
        )

        result, data = await self._send_message(
            request, bonuses_pb2.FetchClientBonusesGraphOutput
        )

        if result == "details":
            return structs.ClientGraph.from_proto(data)
        elif result == "client_not_found":
            raise ClientNotFound(agency_id=agency_id, client_id=client_id)

        raise ProtocolError("Unexpected response")

    async def list_bonuses_reports(self, agency_id: int) -> list[structs.ReportInfo]:
        request = request_pb2.RpcRequest(
            list_bonuses_reports=structs.ListBonusesReportsInfoInput(
                agency_id=agency_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request, reports_pb2.ListBonusesReportsInfoOutput
        )
        if result == "reports":
            return [structs.ReportInfo.from_proto(bonuses_report) for bonuses_report in data.reports]

        raise ProtocolError("Unexpected response")

    async def get_report_url(self, agency_id: int, report_id: int) -> str:
        request = request_pb2.RpcRequest(
            get_report_url=reports_pb2.GetReportUrl(
                agency_id=agency_id,
                report_id=report_id
            )
        )

        result, data = await self._send_message(
            request,
            reports_pb2.GetReportUrlOutput
        )

        if result == 'result':
            return structs.GetReportUrlResponse.from_proto(data).report_url
        elif result == 'no_such_report':
            raise NoSuchReportException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()
        elif result == 'file_not_found':
            raise FileNotFoundException()
        elif result == 'report_not_ready':
            raise ReportNotReadyException()
        raise ProtocolError('Unexpected response')

    async def get_detailed_report_info(self, agency_id: int, report_id: int) -> structs.ReportInfo:
        request = request_pb2.RpcRequest(
            get_detailed_report_info=reports_pb2.GetDetailedReportInfo(
                agency_id=agency_id,
                report_id=report_id
            )
        )
        result, data = await self._send_message(
            request,
            reports_pb2.GetDetailedReportInfoOutput
        )

        if result == 'result':
            return structs.GetDetailedReportInfoResponse.from_proto(data).report
        elif result == 'no_such_report':
            raise NoSuchReportException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()

    async def create_report(
        self,
        agency_id: int,
        name: str,
        period_from: datetime,
        period_to: datetime,
        client_type: structs.ClientType
    ) -> structs.ReportInfo:

        request = request_pb2.RpcRequest(
            create_report=structs.CreateReportInput(
                agency_id=agency_id,
                name=name,
                period_from=period_from,
                period_to=period_to,
                client_type=client_type
            ).to_proto()
        )

        result, data = await self._send_message(
            request, reports_pb2.CreateReportOutput
        )

        if result == 'result':
            return structs.CreateReportOutput.from_proto(data).report

        raise ProtocolError('Unexpected response')

    async def delete_report(self, agency_id: int, report_id: int) -> bool:
        request = request_pb2.RpcRequest(
            delete_report=structs.DeleteReportInput(
                agency_id=agency_id,
                report_id=report_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request, reports_pb2.DeleteReportOutput
        )
        if result == 'result':
            return structs.DeleteReportOutput.from_proto(data).is_deleted
        elif result == 'no_such_report':
            raise NoSuchReportException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()
        raise ProtocolError('Unexpected response')

    async def list_cashback_programs(
        self,
        agency_id: int,
    ) -> list[structs.CashbackProgram]:

        request = request_pb2.RpcRequest(
            list_cashback_programs=structs.ListCashbackProgramsInput(
                agency_id=agency_id,
            ).to_proto()
        )

        result, data = await self._send_message(
            request, bonuses_pb2.ListCashbackProgramsOutput
        )

        if result == "programs":
            return [structs.CashbackProgram.from_proto(program) for program in data.programs]

        raise ProtocolError("Unexpected response")
