from dataclasses import dataclass
from datetime import datetime
from decimal import Decimal
from crm.agency_cabinet.common.enum import BaseEnum
from typing import Optional

from google.protobuf.message import Message
from smb.common.helpers import Converter, PbDatetimeConverter

from crm.agency_cabinet.client_bonuses.proto import bonuses_pb2, common_pb2, reports_pb2
from crm.agency_cabinet.common.consts.report import ReportsStatuses
from crm.agency_cabinet.common.proto_utils import decimal_to_string

__all__ = [
    "BonusAmount",
    "BonusDetails",
    "BonusDetailsList",
    "BonusStatusType",
    "BonusType",
    "ClientBonus",
    "ClientGraph",
    "ClientType",
    "FetchBonusesDetailsInput",
    "GraphPoint",
    "ListClientsBonusesInput",
    "ProgramBonusesGraph",
    "GetClientsBonusesSettingsInput",
    "ClientBonusSettings",
    "ReportInfo",
    "ListBonusesReportsInfoInput",
    "GetDetailedReportInfoResponse",
    "GetReportUrlResponse",
    "CreateReportInput",
    "CreateReportOutput",
    "ListCashbackProgramsInput",
    "CashbackProgram",
]


class BonusStatusType(BaseEnum):
    accrued = "accrued"
    spent = "spent"


class ClientType(BaseEnum):
    ALL = "ALL"
    ACTIVE = "ACTIVE"
    EXCLUDED = "EXCLUDED"


class BonusType(BaseEnum):
    ALL = "ALL"
    WITH_ACTIVATION_OVER_PERIOD = "WITH_ACTIVATION_OVER_PERIOD"
    WITH_SPENDS_OVER_PERIOD = "WITH_SPENDS_OVER_PERIOD"


client_type_converter = Converter(
    [
        (bonuses_pb2.ClientType.ALL_CLIENTS, ClientType.ALL),
        (bonuses_pb2.ClientType.ACTIVE, ClientType.ACTIVE),
        (bonuses_pb2.ClientType.EXCLUDED, ClientType.EXCLUDED),
    ]
)

bonus_type_converter = Converter(
    [
        (bonuses_pb2.BonusType.ALL_BONUSES, BonusType.ALL),
        (
            bonuses_pb2.BonusType.WITH_ACTIVATION_OVER_PERIOD,
            BonusType.WITH_ACTIVATION_OVER_PERIOD,
        ),
        (bonuses_pb2.BonusType.WITH_SPENDS_OVER_PERIOD, BonusType.WITH_SPENDS_OVER_PERIOD),
    ]
)


@dataclass
class ListClientsBonusesInput:
    agency_id: int
    client_type: ClientType
    bonus_type: BonusType
    limit: int
    offset: int
    datetime_start: datetime
    datetime_end: datetime
    search_query: Optional[str] = None

    @classmethod
    def from_proto(
        cls, message: bonuses_pb2.ListClientsBonusesInput
    ) -> "ListClientsBonusesInput":
        return cls(
            agency_id=message.agency_id,
            client_type=client_type_converter.forward(message.client_type),
            bonus_type=bonus_type_converter.forward(message.bonus_type),
            limit=message.limit,
            offset=message.offset,
            datetime_start=PbDatetimeConverter().to_datetime(
                message.period.datetime_start
            ),
            datetime_end=PbDatetimeConverter().to_datetime(message.period.datetime_end),
            search_query=(
                message.search_query if message.HasField("search_query") else None
            ),
        )

    def to_proto(self) -> bonuses_pb2.ListClientsBonusesInput:
        return bonuses_pb2.ListClientsBonusesInput(
            agency_id=self.agency_id,
            client_type=client_type_converter.reversed(self.client_type),
            bonus_type=bonus_type_converter.reversed(self.bonus_type),
            limit=self.limit,
            offset=self.offset,
            period=common_pb2.TimePeriod(
                datetime_start=PbDatetimeConverter().from_datetime(self.datetime_start),
                datetime_end=PbDatetimeConverter().from_datetime(self.datetime_end),
            ),
            search_query=self.search_query,
        )


@dataclass
class ClientBonus:
    client_id: int
    email: str
    active: bool
    currency: str
    accrued: Optional[Decimal] = None
    spent: Optional[Decimal] = None
    awarded: Optional[Decimal] = None

    @classmethod
    def from_proto(cls, message: bonuses_pb2.ClientBonus) -> "ClientBonus":
        return cls(
            client_id=message.client_id,
            email=message.email,
            accrued=Decimal(message.accrued) if message.HasField("accrued") else None,
            spent=Decimal(message.spent) if message.HasField("spent") else None,
            awarded=Decimal(message.awarded) if message.HasField("awarded") else None,
            active=message.active,
            currency=message.currency,
        )

    def to_proto(self) -> bonuses_pb2.ClientBonus:
        return bonuses_pb2.ClientBonus(
            client_id=self.client_id,
            email=self.email,
            accrued=decimal_to_string(self.accrued) if self.accrued is not None else None,
            spent=decimal_to_string(self.spent) if self.spent is not None else None,
            awarded=decimal_to_string(self.awarded) if self.awarded is not None else None,
            active=self.active,
            currency=self.currency,
        )


bonus_status_converter = Converter(
    [
        (bonuses_pb2.BonusStatusType.SPENT, BonusStatusType.spent),
        (bonuses_pb2.BonusStatusType.ACCRUED, BonusStatusType.accrued),
    ]
)


@dataclass
class GetClientsBonusesSettingsInput:
    agency_id: int

    @classmethod
    def from_proto(cls, message: bonuses_pb2.GetClientsBonusesSettingsInput) -> "GetClientsBonusesSettingsInput":
        return cls(agency_id=message.agency_id)

    def to_proto(self) -> bonuses_pb2.GetClientsBonusesSettingsInput:
        return bonuses_pb2.GetClientsBonusesSettingsInput(agency_id=self.agency_id)


@dataclass
class ClientBonusSettings:
    first_date: Optional[datetime] = None
    last_date: Optional[datetime] = None

    @classmethod
    def from_proto(cls, message: bonuses_pb2.ClientBonusSettings) -> "ClientBonusSettings":
        first_date = None
        last_date = None

        if message.first_date.seconds:
            first_date = PbDatetimeConverter().to_datetime(message.first_date)

        if message.last_date.seconds:
            last_date = PbDatetimeConverter().to_datetime(message.last_date)

        return cls(first_date=first_date, last_date=last_date)

    def to_proto(self) -> bonuses_pb2.ClientBonusSettings:
        first_date = None
        last_date = None

        if self.first_date:
            first_date = PbDatetimeConverter().from_datetime(self.first_date)

        if self.last_date:
            last_date = PbDatetimeConverter().from_datetime(self.last_date)

        return bonuses_pb2.ClientBonusSettings(first_date=first_date, last_date=last_date)


@dataclass
class BonusAmount:
    program_id: int
    amount: Decimal

    @classmethod
    def from_proto(cls, message: bonuses_pb2.BonusAmount) -> "BonusAmount":
        return cls(program_id=message.program_id, amount=Decimal(message.amount))

    def to_proto(self) -> Message:
        return bonuses_pb2.BonusAmount(
            program_id=self.program_id, amount=decimal_to_string(self.amount)
        )


@dataclass
class BonusDetails:
    type: BonusStatusType
    date: datetime
    total: Decimal
    amounts: list[BonusAmount]

    @classmethod
    def from_proto(cls, message: bonuses_pb2.BonusAmount) -> "BonusDetails":
        return cls(
            type=bonus_status_converter.forward(message.type),
            total=Decimal(message.total),
            date=PbDatetimeConverter().to_datetime(message.date),
            amounts=[BonusAmount.from_proto(amount) for amount in message.amounts],
        )

    def to_proto(self) -> Message:
        return bonuses_pb2.BonusDetails(
            type=bonus_status_converter.reversed(self.type),
            total=decimal_to_string(self.total),
            date=PbDatetimeConverter().from_datetime(self.date),
            amounts=[amount.to_proto() for amount in self.amounts],
        )


@dataclass
class BonusDetailsList:
    items: list[BonusDetails]

    def to_proto(self) -> Message:
        return bonuses_pb2.BonusDetailsList(
            items=[item.to_proto() for item in self.items]
        )

    @classmethod
    def from_proto(cls, message: bonuses_pb2.BonusDetailsList) -> "BonusDetailsList":
        return cls(items=[BonusDetails.from_proto(item) for item in message.items])


@dataclass
class FetchBonusesDetailsInput:
    agency_id: int
    client_id: int
    datetime_start: datetime
    datetime_end: datetime

    @classmethod
    def from_proto(
        cls, message: bonuses_pb2.FetchBonusesDetailsInput
    ) -> "FetchBonusesDetailsInput":
        return cls(
            agency_id=message.agency_id,
            client_id=message.client_id,
            datetime_start=PbDatetimeConverter().to_datetime(
                message.period.datetime_start
            ),
            datetime_end=PbDatetimeConverter().to_datetime(message.period.datetime_end),
        )

    def to_proto(self) -> Message:
        return bonuses_pb2.FetchBonusesDetailsInput(
            agency_id=self.agency_id,
            client_id=self.client_id,
            period=common_pb2.TimePeriod(
                datetime_start=PbDatetimeConverter().from_datetime(self.datetime_start),
                datetime_end=PbDatetimeConverter().from_datetime(self.datetime_end),
            ),
        )


@dataclass
class GraphPoint:
    point: datetime
    value: Decimal

    @classmethod
    def from_proto(cls, message: bonuses_pb2.GraphPoint) -> "GraphPoint":
        return cls(
            point=PbDatetimeConverter().to_datetime(message.point),
            value=Decimal(message.value),
        )

    def to_proto(self) -> Message:
        return bonuses_pb2.GraphPoint(
            point=PbDatetimeConverter().from_datetime(self.point),
            value=decimal_to_string(self.value),
        )


@dataclass
class ProgramBonusesGraph:
    program_id: int
    historical_monthly_data: list[GraphPoint]

    @classmethod
    def from_proto(
        cls, message: bonuses_pb2.ProgramBonusesGraph
    ) -> "ProgramBonusesGraph":
        return cls(
            program_id=message.program_id,
            historical_monthly_data=[
                GraphPoint.from_proto(monthly_data)
                for monthly_data in message.historical_monthly_data
            ],
        )

    def to_proto(self) -> Message:
        return bonuses_pb2.ProgramBonusesGraph(
            program_id=self.program_id,
            historical_monthly_data=[
                monthly_data.to_proto() for monthly_data in self.historical_monthly_data
            ],
        )


@dataclass
class ClientGraph:
    bonuses_available: Decimal
    overall_spent: list[GraphPoint]
    overall_accrued: list[GraphPoint]
    programs: list[ProgramBonusesGraph]

    @classmethod
    def from_proto(cls, message: bonuses_pb2.ClientGraph) -> "ClientGraph":
        return cls(
            bonuses_available=(
                Decimal(message.bonuses_available)
                if message.bonuses_available
                else None
            ),
            overall_spent=[
                GraphPoint.from_proto(spent) for spent in message.overall_spent
            ],
            overall_accrued=[
                GraphPoint.from_proto(accrued) for accrued in message.overall_accrued
            ],
            programs=[
                ProgramBonusesGraph.from_proto(program) for program in message.programs
            ],
        )

    def to_proto(self):
        return bonuses_pb2.ClientGraph(
            bonuses_available=decimal_to_string(self.bonuses_available),
            overall_spent=[spent.to_proto() for spent in self.overall_spent],
            overall_accrued=[accrued.to_proto() for accrued in self.overall_accrued],
            programs=[program.to_proto() for program in self.programs],
        )


@dataclass
class ReportInfo:
    id: int
    name: str
    period_from: datetime
    period_to: datetime
    created_at: datetime
    client_type: ClientType
    status: str = ReportsStatuses.requested.value

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportInfo) -> 'ReportInfo':
        return cls(
            id=message.id,
            name=message.name,
            status=message.status,
            client_type=client_type_converter.forward(message.client_type),
            created_at=PbDatetimeConverter().to_datetime(message.created_at),
            period_from=PbDatetimeConverter().to_datetime(message.period_from),
            period_to=PbDatetimeConverter().to_datetime(message.period_to),
        )

    def to_proto(self) -> reports_pb2.ReportInfo:
        return reports_pb2.ReportInfo(
            id=self.id,
            name=self.name,
            created_at=PbDatetimeConverter().from_datetime(self.created_at),
            period_from=PbDatetimeConverter().from_datetime(self.period_from),
            period_to=PbDatetimeConverter().from_datetime(self.period_to),
            status=self.status,
            client_type=client_type_converter.reversed(self.client_type),

        )


@dataclass
class ListBonusesReportsInfoInput:
    agency_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.ListBonusesReportsInfoInput) -> 'ListBonusesReportsInfoInput':
        return cls(agency_id=message.agency_id)

    def to_proto(self) -> reports_pb2.ListBonusesReportsInfoInput:
        return reports_pb2.ListBonusesReportsInfoInput(agency_id=self.agency_id)


@dataclass
class GetReportUrlResponse:
    report_url: str

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportUrl) -> 'GetReportUrlResponse':
        return cls(report_url=message.url)

    def to_proto(self) -> reports_pb2.ReportUrl:
        return reports_pb2.ReportUrl(url=self.report_url)


@dataclass
class GetDetailedReportInfoResponse:
    report: ReportInfo

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportInfo) -> 'GetDetailedReportInfoResponse':
        return cls(report=ReportInfo.from_proto(message))

    def to_proto(self) -> reports_pb2.ReportInfo:
        return self.report.to_proto()


@dataclass
class CreateReportInput:
    agency_id: int
    name: str
    period_from: datetime
    period_to: datetime
    client_type: ClientType

    @classmethod
    def from_proto(cls, message: reports_pb2.CreateReportInput) -> 'CreateReportInput':
        return cls(
            agency_id=message.agency_id,
            name=message.name,
            period_from=PbDatetimeConverter().to_datetime(message.period_from),
            period_to=PbDatetimeConverter().to_datetime(message.period_to),
            client_type=client_type_converter.forward(message.client_type)
        )

    def to_proto(self) -> reports_pb2.CreateReportInput:
        return reports_pb2.CreateReportInput(
            agency_id=self.agency_id,
            name=self.name,
            period_from=PbDatetimeConverter().from_datetime(self.period_from),
            period_to=PbDatetimeConverter().from_datetime(self.period_to),
            client_type=client_type_converter.reversed(self.client_type)
        )


@dataclass
class CreateReportOutput:
    report: ReportInfo

    @classmethod
    def from_proto(cls, message: reports_pb2.ReportInfo) -> 'CreateReportOutput':
        return cls(report=ReportInfo.from_proto(message))

    def to_proto(self) -> reports_pb2.ReportInfo:
        return self.report.to_proto()


@dataclass
class DeleteReportInput:
    agency_id: int
    report_id: int

    @classmethod
    def from_proto(cls, message: reports_pb2.DeleteReportInput) -> 'DeleteReportInput':
        return cls(
            agency_id=message.agency_id,
            report_id=message.report_id
        )

    def to_proto(self) -> reports_pb2.DeleteReportInput:
        return reports_pb2.DeleteReportInput(
            agency_id=self.agency_id,
            report_id=self.report_id
        )


@dataclass
class DeleteReportOutput:
    is_deleted: bool

    @classmethod
    def from_proto(cls, message: reports_pb2.DeleteResponse) -> 'DeleteReportOutput':
        return cls(is_deleted=message.is_deleted)

    def to_proto(self) -> reports_pb2.DeleteResponse:
        return reports_pb2.DeleteResponse(is_deleted=self.is_deleted)


@dataclass
class CashbackProgram:
    id: int
    category_id: int
    is_general: bool
    is_enabled: bool
    name_ru: str
    name_en: str
    description_ru: str
    description_en: str

    @classmethod
    def from_proto(cls, message: bonuses_pb2.CashbackProgram) -> "CashbackProgram":
        return cls(
            id=message.id,
            category_id=message.category_id,
            is_general=message.is_general,
            is_enabled=message.is_enabled,
            name_ru=message.name_ru,
            name_en=message.name_en,
            description_ru=message.description_ru,
            description_en=message.description_en,
        )

    def to_proto(self) -> bonuses_pb2.ClientBonus:
        return bonuses_pb2.CashbackProgram(
            id=self.id,
            category_id=self.category_id,
            is_general=self.is_general,
            is_enabled=self.is_enabled,
            name_ru=self.name_ru,
            name_en=self.name_en,
            description_ru=self.description_ru,
            description_en=self.description_en,
        )


@dataclass
class ListCashbackProgramsInput:
    agency_id: int

    @classmethod
    def from_proto(
        cls, message: bonuses_pb2.ListCashbackProgramsInput
    ) -> "ListCashbackProgramsInput":
        return cls(
            agency_id=message.agency_id,
        )

    def to_proto(self) -> bonuses_pb2.ListCashbackProgramsInput:
        return bonuses_pb2.ListCashbackProgramsInput(
            agency_id=self.agency_id,
        )
