import asyncio
import logging.config

from smb.common.multiruntime import io
from smb.common.pgswim import Migrator

from crm.agency_cabinet.client_bonuses.common import QUEUE
from crm.agency_cabinet.client_bonuses.server.lib.db.engine import DB
from crm.agency_cabinet.client_bonuses.server.lib.handler import Handler
from crm.agency_cabinet.common.server.common.config import MdsConfig
from crm.agency_cabinet.common.server.common.logging_config import get_logging_config
from crm.agency_cabinet.common.server.rpc.config import DsnServerConfig
from crm.agency_cabinet.common.service_discovery import ServiceDiscovery

__all__ = [
    "run_server",
    "upgrade_db"
]


io.setup_filesystem("crm/agency_cabinet/client_bonuses/server/")


def run_server():
    cfg = DsnServerConfig.from_environ()
    mds_cfg = MdsConfig.from_environ()
    logging.config.dictConfig(get_logging_config(cfg.verbose))

    asyncio.run(_run_server(cfg, mds_cfg))


async def _run_server(cfg: DsnServerConfig, mds_cfg: MdsConfig):
    db = await DB.create(str(cfg.get_dsn()))

    async with ServiceDiscovery(
        cfg.amqp_url, with_rpc_client=False
    ) as sd:  # type: ServiceDiscovery
        handler = Handler(db, mds_cfg)

        async with sd.rpc_server(QUEUE) as server:
            await handler.setup()
            await server.drain(handler)


def upgrade_db():
    cfg = DsnServerConfig.from_environ()
    logging.config.dictConfig(get_logging_config(cfg.verbose))

    asyncio.run(_upgrade_db(cfg))


async def _upgrade_db(cfg: DsnServerConfig):
    db = await DB.create(str(cfg.get_dsn()))

    migrator = Migrator(db=db, migrations_path="lib/db/migrations")
    await migrator.upgrade()
    await db.close()
