import asyncio

from smb.common.pgswim import PoolType
from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient
from datetime import datetime, timedelta
from crm.agency_cabinet.client_bonuses.server.lib.celery import celery_app
from crm.agency_cabinet.client_bonuses.server.lib.config import BONUSES_LOADER_SETTINGS
from crm.agency_cabinet.client_bonuses.server.lib.db.engine import DB
from crm.agency_cabinet.common.consts.report import ReportsStatuses
from crm.agency_cabinet.common.server.common.config import MdsConfig
from crm.agency_cabinet.common.server.rpc.config import DsnServerConfig
from crm.agency_cabinet.common.celery.base import locked_task

from .load_bonuses_data import BonusesDataLoader, BonusesToActivateDataLoader, BonusesCorrectionsDataLoader, \
    MarketBonusesDataLoader
from .load_cashback_programs import CashbackProgramsLoader
from .generate_report import ReportGenerator

__all__ = [
    "load_bonuses_data_task",
    "check_requested_reports_task",
    "generate_report_task",
    "load_cashback_programs_task",
    "load_bonuses_to_activate_data_task",
    "load_bonuses_corrections_data_task",
    "load_market_bonuses_data_task",
    "clear_faulty_in_progress_reports_task",
    "clear_old_error_reports_task"
]

LOCK_PATH = 'generate_bonuses_report/task'
LOCK_KEY = 'report_id'


@celery_app.task(bind=True)
def load_bonuses_data_task(*args, **kwargs):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_load_bonuses_data(cfg))


async def _load_bonuses_data(cfg: DsnServerConfig):
    db = await DB.create(str(cfg.get_dsn()))
    yt_client = YtClient(
        proxy=BONUSES_LOADER_SETTINGS["yt_proxy"],
        token=BONUSES_LOADER_SETTINGS["yt_token"],
    )
    yql_client = YqlClient(token=BONUSES_LOADER_SETTINGS["yql_token"])

    loader = BonusesDataLoader(db, yt_client, yql_client)

    await loader()


@celery_app.task(bind=True)
def check_requested_reports_task(*args, **kwargs):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_check_requested_reports(cfg))


async def _check_requested_reports(cfg: DsnServerConfig):
    db = await DB.create(str(cfg.get_dsn()))
    async with db.acquire(PoolType.replica) as con:
        reports = await con.fetch(
            f"""
            SELECT id FROM report_meta_info WHERE status='{ReportsStatuses.requested.value}'
            """
        )

    for report in reports:
        generate_report_task.apply_async(
            (),
            {'report_id': report['id']},
            queue='client_bonuses',
        )

    await db.close()


@celery_app.task(bind=True, time_limit=3 * 60 * 60 + 1)
@locked_task(lock_path=LOCK_PATH, key=LOCK_KEY, block_timeout=1, block=True, timeout=3 * 60 * 60)
def generate_report_task(self, report_id):
    cfg = DsnServerConfig.from_environ()
    mds_cfg = MdsConfig.from_environ()
    asyncio.run(_generate_report(cfg, mds_cfg, report_id))


async def _generate_report(cfg: DsnServerConfig, mds_cfg: MdsConfig, report_id: int):
    db = await DB.create(str(cfg.get_dsn()))
    await ReportGenerator(db, mds_cfg, report_id).generate()
    await db.close()


async def _load_cashback_programs(cfg: DsnServerConfig):
    db = await DB.create(cfg.get_dsn())
    yql_client = YqlClient(token=BONUSES_LOADER_SETTINGS["yql_token"])

    loader = CashbackProgramsLoader(db, yql_client)

    await loader()


@celery_app.task(bind=True)
def load_cashback_programs_task(*args, **kwargs):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_load_cashback_programs(cfg))


@celery_app.task(bind=True)
def load_bonuses_to_activate_data_task(*args, **kwargs):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_load_bonuses_to_activate_data_task(cfg))


async def _load_bonuses_to_activate_data_task(cfg: DsnServerConfig):
    db = await DB.create(str(cfg.get_dsn()))
    yt_client = YtClient(
        proxy=BONUSES_LOADER_SETTINGS["yt_proxy"],
        token=BONUSES_LOADER_SETTINGS["yt_token"],
    )
    yql_client = YqlClient(token=BONUSES_LOADER_SETTINGS["yql_token"])

    loader = BonusesToActivateDataLoader(db, yt_client, yql_client)

    await loader()


@celery_app.task(bind=True)
def load_bonuses_corrections_data_task(self, date: str = None):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_load_bonuses_corrections_data_task(cfg, date))


async def _load_bonuses_corrections_data_task(cfg: DsnServerConfig, date: str):
    db = await DB.create(str(cfg.get_dsn()))
    yt_client = YtClient(
        proxy=BONUSES_LOADER_SETTINGS["yt_proxy"],
        token=BONUSES_LOADER_SETTINGS["yt_token"],
    )
    yql_client = YqlClient(token=BONUSES_LOADER_SETTINGS["yql_token"])

    loader = BonusesCorrectionsDataLoader(db, yt_client, yql_client, date)

    await loader()


@celery_app.task(bind=True)
def load_market_bonuses_data_task(self, date: str = None):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_load_market_bonuses_data_task(cfg, date))


async def _load_market_bonuses_data_task(cfg: DsnServerConfig, date: str):
    db = await DB.create(str(cfg.get_dsn()))
    yt_client = YtClient(
        proxy=BONUSES_LOADER_SETTINGS["yt_proxy"],
        token=BONUSES_LOADER_SETTINGS["yt_token"],
    )
    yql_client = YqlClient(token=BONUSES_LOADER_SETTINGS["yql_token"])

    loader = MarketBonusesDataLoader(db, yt_client, yql_client, date)

    await loader()


async def _clear_faulty_in_progress_reports_task(cfg, delta_hours: int, lock_manager):
    db = await DB.create(str(cfg.get_dsn()))
    now = datetime.now()
    diff = now - timedelta(hours=delta_hours)

    async with db.acquire(PoolType.replica) as con:
        reports = await con.fetch(
            f"""
                SELECT id FROM report_meta_info
                WHERE
                    status='{ReportsStatuses.requested.value}'
                    AND
                    updated_at < '{diff.isoformat()}'
                """
        )

    for report in reports:
        path = f"{LOCK_PATH}/{LOCK_KEY}-{report['id']}"

        lock = lock_manager.lock(path, block=False, block_timeout=0, timeout=1)
        with lock.acquire(1) as have_lock:
            if have_lock:
                async with db.acquire(PoolType.master) as con:
                    await con.execute(
                        f"""
                            UPDATE report_meta_info
                            SET STATUS='{ReportsStatuses.error.value}', UPDATED_AT = NOW()
                            WHERE
                                id={report['id']}
                            """
                    )
    await db.close()


@celery_app.task(bind=True)
def clear_faulty_in_progress_reports_task(self, delta_hours=1):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_clear_faulty_in_progress_reports_task(cfg, delta_hours=delta_hours, lock_manager=self.LOCK_MANAGER))


async def _clear_old_error_reports_task(cfg, delta_hours: int):
    db = await DB.create(str(cfg.get_dsn()))
    now = datetime.now()
    diff = now - timedelta(hours=delta_hours)

    async with db.acquire(PoolType.master) as con:
        await con.execute(
            f"""
                DELETE FROM report_meta_info
                WHERE
                    status='{ReportsStatuses.error.value}'
                    AND
                    updated_at < '{diff.isoformat()}'
                """
        )

    await db.close()


@celery_app.task(bind=True)
def clear_old_error_reports_task(self, delta_hours=24 * 3):
    cfg = DsnServerConfig.from_environ()
    asyncio.run(_clear_old_error_reports_task(cfg, delta_hours=delta_hours))
