from datetime import datetime, timedelta, timezone
from typing import Optional

from smb.common.pgswim import PoolType, SwimEngine
from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder
from yt.wrapper import YtClient, ypath_join

from .synchronizers import (
    ClientBonusesToActivateTableSynchronizer,
    ClientsProgramsTableSynchronizer,
    ClientsTableSynchronizer,
    GainedClientBonusesTableSynchronizer,
    SpentClientBonusesTableSynchronizer,
)


class BonusesDataLoader:
    gained_bonuses_root_old = (
        "//home/balance/prod/yb-ar/cashback/direct/bonus-cashback-2020"  # earliest table - 202103
    )
    gained_bonuses_root_new = (
        "//home/balance/prod/yb-ar/rewards/direct/2021-10-direct-features"  # earliest table - 202110
    )
    spent_bonuses_root = "//home/agency_analytics/bonuses/funnel/spent"
    direct_dump_root = "//home/direct/mysql-sync/current"

    # Earliest table with good format
    EARLIEST_DATE = datetime(2021, 3, 1, tzinfo=timezone.utc)

    NEW_GAINED_TABLE_DATE = datetime(2021, 10, 1, tzinfo=timezone.utc)

    def __init__(self, db: SwimEngine, yt_client: YtClient, yql_client: YqlClient):
        self._db = db
        self._yt_client = yt_client
        self._yql_client = yql_client

    async def __call__(self) -> None:
        date_to_load = await self._get_get_date_to_load()
        if date_to_load is None:
            return

        request = self._yql_client.query(YQL_BONUSES)

        parameters = {
            "$gained_bonuses_root": YqlParameterValueBuilder.make_string(
                self.gained_bonuses_root
            ),
            "$spent_bonuses_root": YqlParameterValueBuilder.make_string(
                self.spent_bonuses_root
            ),
            "$direct_dump_root": YqlParameterValueBuilder.make_string(
                self.direct_dump_root
            ),
            "$dt": YqlParameterValueBuilder.make_string(
                date_to_load.strftime("%Y-%m-%d")
            ),
        }
        request.run(parameters=YqlParameterValueBuilder.build_json_map(parameters))
        results = request.get_results()

        if results.status == "ERROR":
            raise Exception(f"YQL failed: {results.text}")

        (
            client,
            client_program,
            gained_client_bonuses,
            spent_client_bonuses,
        ) = list(result.get_iterator() for result in results)

        async with self._db.acquire(PoolType.master) as con:
            async with con.transaction():
                await ClientsTableSynchronizer(con).process_data(client)
                await ClientsProgramsTableSynchronizer(con).process_data(client_program)
                await GainedClientBonusesTableSynchronizer(
                    con, date_to_load
                ).process_data(gained_client_bonuses)
                await SpentClientBonusesTableSynchronizer(
                    con, date_to_load
                ).process_data(spent_client_bonuses)

    async def _get_get_date_to_load(self) -> Optional[datetime]:
        async with self._db.acquire(PoolType.replica) as con:
            # Find latest date for loaded data
            latest_date_with_data = await con.fetchval(
                """
                SELECT greatest(
                    (SELECT max(gained_at AT TIME ZONE 'utc') FROM gained_client_bonuses),
                    (SELECT max(spent_at AT TIME ZONE 'utc') FROM spent_client_bonuses)
                )
                """  # noqa
            )

        if latest_date_with_data is not None:
            latest_date_with_data = latest_date_with_data.replace(tzinfo=timezone.utc)
            date_to_load = (
                latest_date_with_data.replace(day=1, hour=0, minute=0, second=0)
                + timedelta(days=32)
            ).replace(day=1)
        else:
            date_to_load = self.EARLIEST_DATE

        # Check if new data can be available to date
        now = datetime.now(tz=timezone.utc)
        if (
            date_to_load.year > now.year
            or date_to_load.year == now.year
            and date_to_load.month >= now.month
        ):
            return None

        if date_to_load >= self.NEW_GAINED_TABLE_DATE:
            self.gained_bonuses_root = self.gained_bonuses_root_new
        else:
            self.gained_bonuses_root = self.gained_bonuses_root_old

        # Check if new data is actually available
        if not self._yt_client.exists(
            ypath_join(self.gained_bonuses_root, date_to_load.strftime("%Y%m"))
        ):
            return None
        if not self._yt_client.exists(
            ypath_join(self.spent_bonuses_root, date_to_load.strftime("%Y-%m-%d"))
        ):
            return None

        return date_to_load


YQL_BONUSES = r"""

USE hahn;

DECLARE $gained_bonuses_root AS String;
DECLARE $spent_bonuses_root AS String;
DECLARE $direct_dump_root AS String;
DECLARE $dt AS String;

$dt_short = Re2::Replace(@@(\d{4})-(\d{2})-(\d{2})@@)($dt, "\\1\\2");

$clients = (
    SELECT
        clients.ClientID AS id,
        COALESCE(MAX(users.login), "") AS login,
        MAX(agency_client_relations.agency_client_id) AS agency_id,
        bool_and(agency_client_relations.client_archived = 'No') as is_active,
        MAX(clients.create_date) as create_date
    FROM LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/clients`
    ) AS clients
    LEFT JOIN LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/agency_client_relations`
    ) AS agency_client_relations
        ON clients.ClientID = agency_client_relations.client_client_id
    LEFT JOIN LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/users`
    ) AS users
        ON clients.chief_uid = users.uid
    WHERE agency_client_relations.agency_client_id IS NOT NULL
    GROUP BY clients.ClientID
);

$clients_programs = (
    SELECT
        clients.id AS client_id,
        cashback_program_id AS program_id
        --cashback_program_id AS program_id
    FROM LIKE (
        $direct_dump_root,
        `ppc:%`,
        `straight/clients_cashback_programs`
    ) AS client_programs
    JOIN $clients AS clients
        ON client_programs.ClientID = clients.id
    WHERE client_programs.is_enabled = 1
        AND cashback_program_id IS NOT NULL
);

$gained_client_bonuses = (
    SELECT
        gained.client_id AS client_id,
        program_id,
        reward AS amount,
        currency AS currency
    FROM (
        SELECT
            client_id,
            Yson::ConvertToInt64(detail.program_id) AS program_id,
            CAST(Yson::LookupDouble(detail, "reward") AS String) AS reward,
            currency
        FROM (
            SELECT
                client_id,
                Yson::ConvertToList(Yson::ParseJson(cashback_details).details) AS details,
                currency
            FROM LIKE($gained_bonuses_root, $dt_short)
        )
        FLATTEN LIST BY (details AS detail)
        WHERE Yson::ConvertToDouble(detail.reward) > 0
    ) AS gained
    JOIN $clients AS clients
        ON gained.client_id = clients.id
    WHERE program_id IS NOT NULL
);

$spent_client_bonuses = (
    SELECT
        spent.client_id AS client_id,
        CAST(spent_bonus_w_nds AS String) AS amount,
        currency AS currency
    FROM LIKE($spent_bonuses_root, $dt) AS spent
    JOIN $clients AS clients
        ON spent.client_id = clients.id
);

SELECT id, login, agency_id, is_active, create_date
FROM $clients
ORDER BY id;

SELECT client_id, program_id
FROM $clients_programs
ORDER BY client_id;

SELECT client_id, program_id, amount, currency
FROM $gained_client_bonuses
ORDER BY client_id, program_id;

SELECT client_id, amount, currency
FROM $spent_client_bonuses
ORDER BY client_id;
"""  # noqa


class BonusesToActivateDataLoader:
    direct_dump_root = "//home/direct/mysql-sync/current"

    def __init__(self, db: SwimEngine, yt_client: YtClient, yql_client: YqlClient):
        self._db = db
        self._yt_client = yt_client
        self._yql_client = yql_client

    async def __call__(self) -> None:
        request = self._yql_client.query(YQL_BONUSES_TO_ACTIVATE)

        parameters = {
            "$direct_dump_root": YqlParameterValueBuilder.make_string(
                self.direct_dump_root
            ),
        }
        request.run(parameters=YqlParameterValueBuilder.build_json_map(parameters))
        results = request.get_results()

        if results.status == "ERROR":
            raise Exception(f"YQL failed: {results.text}")

        client_bonuses_to_activate = results.table.get_iterator()

        async with self._db.acquire(PoolType.master) as con:
            async with con.transaction():
                await ClientBonusesToActivateTableSynchronizer(con).process_data(
                    client_bonuses_to_activate
                )


YQL_BONUSES_TO_ACTIVATE = """
USE hahn;

DECLARE $direct_dump_root AS String;

$clients = (
    SELECT
        clients.ClientID AS id,
        COALESCE(MAX(users.login), "") AS login,
        MAX(agency_client_relations.agency_client_id) AS agency_id,
        bool_and(agency_client_relations.client_archived = 'No') as is_active,
        MAX(clients.create_date) as create_date
    FROM LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/clients`
    ) AS clients
    LEFT JOIN LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/agency_client_relations`
    ) AS agency_client_relations
        ON clients.ClientID = agency_client_relations.client_client_id
    LEFT JOIN LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/users`
    ) AS users
        ON clients.chief_uid = users.uid
    WHERE agency_client_relations.agency_client_id IS NOT NULL
    GROUP BY clients.ClientID
);

$client_bonuses_to_activate = (
    SELECT
        options.ClientID AS client_id,
        options.cashback_awaiting_bonus AS amount
    FROM LIKE(
        $direct_dump_root,
        "ppc:%",
        "straight/clients_options"
    ) AS options
    JOIN $clients AS clients
        ON options.ClientID = clients.id
    WHERE options.cashback_awaiting_bonus IS NOT NULL
);

SELECT client_id, amount
FROM $client_bonuses_to_activate;
"""  # noqa


class BonusesCorrectionsDataLoader:
    gained_bonuses_corrections_root = (
        "//home/projects/cashback_ns/direct/bonus-cashback-2021-ns"
    )
    direct_dump_root = "//home/direct/mysql-sync/current"

    def __init__(self, db: SwimEngine, yt_client: YtClient, yql_client: YqlClient, date_to_load: str):
        self._db = db
        self._yt_client = yt_client
        self._yql_client = yql_client
        self._date_to_load = datetime.strptime(date_to_load, "%Y%m") if date_to_load else (
            datetime.now() - timedelta(days=32)
        ).replace(day=1,  hour=0, minute=0, second=0, microsecond=0, tzinfo=timezone.utc)

    async def __call__(self) -> None:
        request = self._yql_client.query(YQL_CORRECTIONS_BONUSES)

        parameters = {
            "$direct_dump_root": YqlParameterValueBuilder.make_string(
                self.direct_dump_root
            ),
            "$gained_bonuses_corrections_root": YqlParameterValueBuilder.make_string(
                self.gained_bonuses_corrections_root
            ),
            "$dt_short": YqlParameterValueBuilder.make_string(
                self._date_to_load.strftime("%Y%m")
            ),
        }
        request.run(parameters=YqlParameterValueBuilder.build_json_map(parameters))
        results = request.get_results()

        if results.status == "ERROR":
            raise Exception(f"YQL failed: {results.text}")

        gained_bonuses_corrections = results.table.get_iterator()

        async with self._db.acquire(PoolType.master) as con:
            async with con.transaction():
                await GainedClientBonusesTableSynchronizer(
                    con, self._date_to_load, program_ids=[14]
                ).process_data(gained_bonuses_corrections)


YQL_CORRECTIONS_BONUSES = """
USE hahn;

DECLARE $direct_dump_root AS String;
DECLARE $gained_bonuses_corrections_root AS String;
DECLARE $dt_short AS String;

$clients = (
    SELECT
        clients.ClientID AS id,
        COALESCE(MAX(users.login), "") AS login,
        MAX(agency_client_relations.agency_client_id) AS agency_id,
        bool_and(agency_client_relations.client_archived = 'No') as is_active,
        MAX(clients.create_date) as create_date
    FROM LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/clients`
    ) AS clients
    LEFT JOIN LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/agency_client_relations`
    ) AS agency_client_relations
        ON clients.ClientID = agency_client_relations.client_client_id
    LEFT JOIN LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/users`
    ) AS users
        ON clients.chief_uid = users.uid
    WHERE agency_client_relations.agency_client_id IS NOT NULL
    GROUP BY clients.ClientID
);

$gained_client_bonuses_corrections = (
    SELECT
        gained.client_id AS client_id,
        14 as program_id,
        reward AS amount,
        currency AS currency
    FROM LIKE($gained_bonuses_corrections_root, $dt_short)
    AS gained
    JOIN $clients AS clients
        ON gained.client_id = clients.id
);

SELECT client_id, program_id, amount, currency
FROM $gained_client_bonuses_corrections
ORDER BY client_id, program_id;
"""  # noqa


class MarketBonusesDataLoader:
    market_bonuses_root = (
        "//home/agency_analytics/bonuses/market"  # earliest table - 202106
    )
    direct_dump_root = "//home/direct/mysql-sync/current"

    def __init__(self, db: SwimEngine, yt_client: YtClient, yql_client: YqlClient, date_to_load: str):
        self._db = db
        self._yt_client = yt_client
        self._yql_client = yql_client
        self._date_to_load = datetime.strptime(date_to_load, "%Y%m") if date_to_load else (
            datetime.now() - timedelta(days=32)
        ).replace(day=1,  hour=0, minute=0, second=0, microsecond=0, tzinfo=timezone.utc)

    async def __call__(self) -> None:
        request = self._yql_client.query(YQL_MARKET_BONUSES)

        parameters = {
            "$direct_dump_root": YqlParameterValueBuilder.make_string(
                self.direct_dump_root
            ),
            "$market_bonuses_root": YqlParameterValueBuilder.make_string(
                self.market_bonuses_root
            ),
            "$dt_short": YqlParameterValueBuilder.make_string(
                self._date_to_load.strftime("%Y%m")
            ),
        }
        request.run(parameters=YqlParameterValueBuilder.build_json_map(parameters))
        results = request.get_results()

        if results.status == "ERROR":
            raise Exception(f"YQL failed: {results.text}")

        market_bonuses = results.table.get_iterator()

        async with self._db.acquire(PoolType.master) as con:
            async with con.transaction():
                await GainedClientBonusesTableSynchronizer(
                    con, self._date_to_load, program_ids=[7, 13]
                ).process_data(market_bonuses)


YQL_MARKET_BONUSES = """
USE hahn;

DECLARE $direct_dump_root AS String;
DECLARE $market_bonuses_root AS String;
DECLARE $dt_short AS String;

$clients = (
    SELECT
        clients.ClientID AS id,
        COALESCE(MAX(users.login), "") AS login,
        MAX(agency_client_relations.agency_client_id) AS agency_id,
        bool_and(agency_client_relations.client_archived = 'No') as is_active,
        MAX(clients.create_date) as create_date
    FROM LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/clients`
    ) AS clients
    LEFT JOIN LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/agency_client_relations`
    ) AS agency_client_relations
        ON clients.ClientID = agency_client_relations.client_client_id
    LEFT JOIN LIKE(
        $direct_dump_root,
        `ppc:%`,
        `straight/users`
    ) AS users
        ON clients.chief_uid = users.uid
    WHERE agency_client_relations.agency_client_id IS NOT NULL
    GROUP BY clients.ClientID
);

$gained_client_bonuses_market = (
    SELECT
        gained.client_id AS client_id,
        program_id,
        reward AS amount,
        currency AS currency
    FROM (
        SELECT
            client_id,
            Yson::ConvertToInt64(detail.program_id) AS program_id,
            CAST(Yson::LookupDouble(detail, "reward") AS String) AS reward,
            currency
        FROM (
            SELECT
                client_id,
                Yson::ConvertToList(Yson::ParseJson(cashback_details).details) AS details,
                currency
            FROM LIKE($market_bonuses_root, $dt_short)
        )
        FLATTEN LIST BY (details AS detail)
        WHERE Yson::ConvertToDouble(detail.reward) > 0
    ) AS gained
    JOIN $clients AS clients
        ON gained.client_id = clients.id
    WHERE program_id IS NOT NULL
);

SELECT client_id, program_id, amount, currency
FROM $gained_client_bonuses_market
ORDER BY client_id, program_id;
"""  # noqa
