from yql.api.v1.client import YqlClient
from yql.client.parameter_value_builder import YqlParameterValueBuilder

from smb.common.pgswim import PoolType, SwimEngine

from .synchronizers import CashbackProgramsSynchronizer


class CashbackProgramsLoader:
    cashback_categories_table = "//home/direct/mysql-sync/current/ppcdict/straight/cashback_categories"
    cashback_programs_table = "//home/direct/mysql-sync/current/ppcdict/straight/cashback_programs"

    _yql = r"""
        USE hahn;

        DECLARE $cashback_categories_table AS String;
        DECLARE $cashback_programs_table AS String;

        SELECT
            `cashback_program_id`,
            categories.`cashback_category_id`,
            `is_general`,
            `is_enabled`,
            categories.`name_ru`,
            categories.`name_en`,
            categories.`description_ru`,
            categories.`description_en`
        FROM $cashback_programs_table AS programs
        INNER JOIN
             $cashback_categories_table AS categories
        ON programs.cashback_category_id = categories.cashback_category_id;
    """

    def __init__(self, db: SwimEngine, yql_client: YqlClient):
        self._db = db
        self._yql_client = yql_client

    async def __call__(self) -> None:
        request = self._yql_client.query(self._yql)

        parameters = {
            "$cashback_categories_table": YqlParameterValueBuilder.make_string(
                self.cashback_categories_table
            ),
            "$cashback_programs_table": YqlParameterValueBuilder.make_string(
                self.cashback_programs_table
            ),
        }
        request.run(parameters=YqlParameterValueBuilder.build_json_map(parameters))
        results = request.get_results()

        if results.status == "ERROR":
            raise Exception(f"YQL failed: {results.text}")

        cashback_programs = results.table.get_iterator()

        async with self._db.acquire(PoolType.master) as con:
            async with con.transaction():
                await CashbackProgramsSynchronizer(con).process_data(
                    cashback_programs
                )
