import aioboto3
import logging
from contextlib import AsyncExitStack
from aiobotocore.config import AioConfig

from smb.common.rmq.rpc.server import BaseRpcHandler

from crm.agency_cabinet.client_bonuses.common.structs import (
    FetchBonusesDetailsInput,
    ListClientsBonusesInput,
    GetClientsBonusesSettingsInput,
    ListBonusesReportsInfoInput,
    CreateReportInput,
    DeleteReportInput,
    ListCashbackProgramsInput
)
from crm.agency_cabinet.client_bonuses.proto import (
    bonuses_pb2,
    common_pb2,
    errors_pb2,
    request_pb2,
    reports_pb2,
)
from crm.agency_cabinet.client_bonuses.server.lib.db.engine import DB
from crm.agency_cabinet.common.server.common.config import MdsConfig
from .exceptions import (
    ClientNotFound, FileNotFoundException, UnsuitableAgencyException,
    ReportNotReadyException, NoSuchReportException
)
from .procedures import (
    FetchBonusesDetails, FetchClientBonusesGraph, ListClientsBonuses, GetClientsBonusesSettings,
    GetReportUrl, GetDetailedReportInfo, ListBonusesReportsInfo, CreateReport, DeleteReport,
    ListCashbackPrograms
)

LOGGER = logging.getLogger('client_bonuses.handler')


class Handler(BaseRpcHandler):
    _request_proto = request_pb2.RpcRequest

    _list_clients_cru: ListClientsBonuses

    def __init__(self, db: DB, mds_cfg: MdsConfig):
        self._fetch_bonuses_details_proc = FetchBonusesDetails(db=db)
        self._list_clients_bonuses_proc = ListClientsBonuses(db=db)
        self._fetch_client_bonuses_graph = FetchClientBonusesGraph(db=db)
        self._get_clients_bonuses_settings_proc = GetClientsBonusesSettings(db=db)
        self._list_bonuses_reports = ListBonusesReportsInfo(db=db)
        self._get_report_url = GetReportUrl(db=db)
        self._get_detailed_report_info = GetDetailedReportInfo(db=db)
        self._mds_cfg = mds_cfg
        self._create_report = CreateReport(db=db)
        self._delete_report = DeleteReport(db=db)
        self._list_cashback_programs = ListCashbackPrograms(db=db)

    async def setup(self):
        success = await self._setup_s3()
        LOGGER.info('Handler successfully init' if success else 'Failed to fully init handler')
        return success

    async def _setup_s3(self) -> bool:
        self.boto3_session = aioboto3.Session(
            aws_access_key_id=self._mds_cfg.access_key_id,
            aws_secret_access_key=self._mds_cfg.secret_access_key
        )
        self.context_stack = AsyncExitStack()
        self.s3_resource = await self.context_stack.enter_async_context(
            self.boto3_session.resource(
                's3',
                endpoint_url=self._mds_cfg.external_endpoint_url,
                config=AioConfig(s3={'addressing_style': 'virtual'})
            )
        )
        self.s3_client = await self.context_stack.enter_async_context(
            self.boto3_session.client(
                's3',
                endpoint_url=self._mds_cfg.external_endpoint_url,
                config=AioConfig(s3={'addressing_style': 'virtual'})
            )
        )
        return True

    async def ping(self, _: common_pb2.Empty) -> common_pb2.PingOutput:
        return common_pb2.PingOutput(ping="pong")

    async def list_clients_bonuses(
        self, message: bonuses_pb2.ListClientsBonusesInput
    ) -> bonuses_pb2.ListClientsBonusesOutput:
        params = ListClientsBonusesInput.from_proto(message)

        clients_bonuses = await self._list_clients_bonuses_proc(params=params)

        return bonuses_pb2.ListClientsBonusesOutput(
            bonuses=bonuses_pb2.ClientsBonusesList(
                bonuses=[bonus.to_proto() for bonus in clients_bonuses]
            )
        )

    async def get_clients_bonuses_settings(
        self, message: bonuses_pb2.GetClientsBonusesSettingsInput
    ) -> bonuses_pb2.GetClientsBonusesSettingsOutput:
        params = GetClientsBonusesSettingsInput.from_proto(message)

        clients_bonuses_settings = await self._get_clients_bonuses_settings_proc(params=params)

        return bonuses_pb2.GetClientsBonusesSettingsOutput(
            settings=clients_bonuses_settings.to_proto()
        )

    async def fetch_bonuses_details(
        self, message: bonuses_pb2.FetchBonusesDetailsInput
    ) -> bonuses_pb2.FetchBonusesDetailsOutput:
        input_params = FetchBonusesDetailsInput.from_proto(message)

        try:
            bonuses_details = await self._fetch_bonuses_details_proc(
                params=input_params
            )

            return bonuses_pb2.FetchBonusesDetailsOutput(
                details=bonuses_details.to_proto()
            )

        except ClientNotFound as exc:
            return bonuses_pb2.FetchBonusesDetailsOutput(
                client_not_found=errors_pb2.ClientNotFound(
                    client_id=exc.client_id, agency_id=exc.agency_id
                )
            )

    async def fetch_client_bonuses_graph(
        self, message: bonuses_pb2.FetchClientBonusesGraphInput
    ) -> bonuses_pb2.FetchClientBonusesGraphOutput:

        try:
            client_bonuses_graph = await self._fetch_client_bonuses_graph(
                client_id=message.client_id, agency_id=message.agency_id
            )

            return bonuses_pb2.FetchClientBonusesGraphOutput(
                details=client_bonuses_graph.to_proto()
            )

        except ClientNotFound as exc:
            return bonuses_pb2.FetchClientBonusesGraphOutput(
                client_not_found=errors_pb2.ClientNotFound(
                    client_id=exc.client_id, agency_id=exc.agency_id
                )
            )

    async def list_bonuses_reports(
        self, message: reports_pb2.ListBonusesReportsInfoInput
    ) -> reports_pb2.ListBonusesReportsInfoOutput:

        params = ListBonusesReportsInfoInput.from_proto(message)
        bonuses_reports_list = await self._list_bonuses_reports(params=params)

        return reports_pb2.ListBonusesReportsInfoOutput(
            reports=reports_pb2.BonusesReportsInfoList(
                reports=[bonuses_report.to_proto() for bonuses_report in bonuses_reports_list]
            )
        )

    async def get_report_url(self, message: reports_pb2.GetReportUrl) -> reports_pb2.GetReportUrlOutput:
        try:
            result = await self._get_report_url(
                agency_id=message.agency_id,
                report_id=message.report_id,
                s3_client=self.s3_client
            )
            return reports_pb2.GetReportUrlOutput(result=result.to_proto())
        except NoSuchReportException:
            return reports_pb2.GetReportUrlOutput(no_such_report=common_pb2.Empty())
        except UnsuitableAgencyException:
            return reports_pb2.GetReportUrlOutput(unsuitable_agency=common_pb2.Empty())
        except FileNotFoundException:
            return reports_pb2.GetReportUrlOutput(file_not_found=common_pb2.Empty())
        except ReportNotReadyException:
            return reports_pb2.GetReportUrlOutput(report_not_ready=common_pb2.Empty())

    async def get_detailed_report_info(
        self, message: reports_pb2.GetDetailedReportInfo
    ) -> reports_pb2.GetDetailedReportInfoOutput:
        try:
            result = await self._get_detailed_report_info(
                agency_id=message.agency_id,
                report_id=message.report_id
            )
            return reports_pb2.GetDetailedReportInfoOutput(result=result.to_proto())
        except NoSuchReportException:
            return reports_pb2.GetDetailedReportInfoOutput(no_such_report=common_pb2.Empty())
        except UnsuitableAgencyException:
            return reports_pb2.GetDetailedReportInfoOutput(unsuitable_agency=common_pb2.Empty())

    async def create_report(
        self, message: reports_pb2.CreateReportInput
    ) -> reports_pb2.CreateReportOutput:

        params = CreateReportInput.from_proto(message)
        result = await self._create_report(params=params)
        return reports_pb2.CreateReportOutput(result=result.to_proto())

    async def delete_report(
        self, message: reports_pb2.DeleteReportInput
    ) -> reports_pb2.DeleteReportOutput:
        try:
            params = DeleteReportInput.from_proto(message)
            result = await self._delete_report(params=params)
            return reports_pb2.DeleteReportOutput(result=result.to_proto())
        except NoSuchReportException:
            return reports_pb2.DeleteReportOutput(no_such_report=common_pb2.Empty())
        except UnsuitableAgencyException:
            return reports_pb2.DeleteReportOutput(unsuitable_agency=common_pb2.Empty())

    async def list_cashback_programs(
        self, message: bonuses_pb2.ListCashbackProgramsInput
    ) -> bonuses_pb2.ListCashbackProgramsOutput:

        params = ListCashbackProgramsInput.from_proto(message)
        programs = await self._list_cashback_programs(params=params)

        return bonuses_pb2.ListCashbackProgramsOutput(
            programs=bonuses_pb2.CashbackProgramsList(
                programs=[program.to_proto() for program in programs]
            )
        )
