import blackbox
import logging

from crm.agency_cabinet.common.server.common.tvm import get_tvm_client
from crm.agency_cabinet.common.server.common.tvm import Tvm2Config
from .config import BlackboxConfig

LOGGER = logging.getLogger('blackbox_client')


class BlackboxClientException(Exception):
    pass


class BlackboxClient:
    def __init__(self, tvm_cfg: Tvm2Config, blackbox_cfg: BlackboxConfig):
        self.tvm_cfg = tvm_cfg
        self.blackbox_cfg = blackbox_cfg

    async def get_uid_by_email(self, email: str, real_ip: str) -> int:
        try:
            blackbox_instance = blackbox.Blackbox(self.blackbox_cfg.url)

            ticket = await get_tvm_client(self.tvm_cfg).get_service_ticket(self.blackbox_cfg.tvm_id)

            response = blackbox_instance.userinfo(
                uid_or_login=email,
                by_login=True,
                userip=real_ip,
                headers={'X-Ya-Service-Ticket': ticket}
            )

            return int(response.get('uid'))
        except Exception as ex:
            LOGGER.exception("Something went wrong: %s", ex)
            raise BlackboxClientException(ex)

    async def get_email_by_uid(self, uid: int, real_ip: str) -> str:
        try:
            blackbox_instance = blackbox.Blackbox(self.blackbox_cfg.url)

            ticket = await get_tvm_client(self.tvm_cfg).get_service_ticket(self.blackbox_cfg.tvm_id)

            response = blackbox_instance.userinfo(
                uid_or_login=uid,
                by_login=False,
                userip=real_ip,
                headers={'X-Ya-Service-Ticket': ticket},
                dbfields=[blackbox.FIELD_EMAIL, blackbox.FIELD_LOGIN]
            )

            login = response.get('fields').get('login')
            email = response.get('fields').get('email')

            return email or (login + '@yandex.ru' if login else None)
        except Exception as ex:
            LOGGER.exception("Something went wrong: %s", ex)
            raise BlackboxClientException(ex)

    async def get_app_client_id_by_oauth_token(self, oauth_token: str, real_ip: str) -> str:
        try:
            blackbox_instance = blackbox.Blackbox(self.blackbox_cfg.url)

            ticket = await get_tvm_client(self.tvm_cfg).get_service_ticket(self.blackbox_cfg.tvm_id)

            response = blackbox_instance.oauth(
                headers_or_token=oauth_token,
                by_token=True,
                userip=real_ip,
                headers={'X-Ya-Service-Ticket': ticket},
            )
            app_client_id = response.get('oauth').get('client_id')
            return app_client_id
        except Exception as ex:
            LOGGER.exception("Something went wrong: %s", ex)
            raise BlackboxClientException(ex)
