import logging
import re
import typing
from aiohttp import ClientSession, ClientError, client

LOGGER = logging.getLogger('bunker.client')


class BunkerError(Exception):
    node: str
    version: str
    message: str

    def __init__(self, message: str, node: str = None, version: str = None):
        self.node = node
        self.version = version
        self.message = message

    def __str__(self):
        return f'Node: \'{self.node}\', version: \'{self.version}\', error message: \'{self.message}\''


class BunkerNotFoundError(BunkerError):
    pass


class BunkerClient:
    TIMEOUT = 2

    def __init__(
        self,
        project: str,
        host: str = 'bunker-api-dot.yandex.net',
        default_version: str = 'stable',
    ):
        self.project = project
        self.host = host
        self.default_version = default_version

        self._root = (
            '/'
            if project is None
            else f'/{project}/'
        )
        self._url = f'http://{host}/v1'

    @staticmethod
    def _strip_url(url: str) -> str:
        return url.strip().strip('/')

    @staticmethod
    def _extract_version(version: typing.Optional[str]) -> str:
        if version is None:
            LOGGER.exception('Invalid version: \'%s\', version should not be None', version)
            raise BunkerError(f'Invalid version: \'{version}\', version should not be None')

        version = re.sub('\\D', '', version)

        if int(version) < 1:
            LOGGER.exception('Invalid version: \'%s\', version should be >= 1', version)
            raise BunkerError(f'Invalid version: \'{version}\', version should be >= 1')

        return version

    async def _make_request(self, command: str, node: str, project: str = None, version: str = None) -> client.ClientResponse:
        url = f'{self._url}/{self._strip_url(command)}'

        node = node.replace(self._root, '')
        node = self._strip_url(node)
        if project is not None:
            node = f'/{self._strip_url(project)}/{node}'
        else:
            node = self._root + node

        if version is None:
            version = self.default_version or 'stable'

        LOGGER.debug('Making request with following parameters: '
                     'URL - \'%s\', node - \'%s\', version - \'%s\'', url, node, version)

        async with ClientSession() as session:
            try:
                async with session.get(
                    url,
                    params={
                        'node': node,
                        'version': version
                    },
                    timeout=self.TIMEOUT,
                ) as response:
                    # TODO: retry policy
                    await response.read()
                    return response
            except ClientError as ex:
                LOGGER.exception('Something went wrong: %s', ex)
                raise BunkerError(node=node, version=version, message=f'Something went wrong: {str(ex)}')

    # TODO: caching?
    async def cat(self, node: str, project: str = None, version: typing.Union[str, int] = None) -> typing.Dict:
        if version is not None:
            version = str(version)
        else:
            version = self.default_version or 'stable'

        data = await self._make_request(command='cat', node=node, project=project, version=version)
        LOGGER.debug(f'Result data: {data}')

        if data.status not in (200, 404):
            LOGGER.exception('Something went wrong with Bunker: %s', str(data))
            raise BunkerError(node=node, version=version, message=f'Something went wrong with Bunker: {str(data)}')

        if data.status == 404:
            LOGGER.exception(
                'Node \'%s\' with version \'%s\' was not found in project \'%s\'', node, version, self.project
            )
            raise BunkerNotFoundError(
                node=node,
                version=version,
                message=f'Node \'{node}\' with version \'{version}\' was not found in project \'{self.project}\''
            )

        return {
            'data': await data.json(),
            'version': self._extract_version(data.headers.get('Etag'))
        }
