import logging
import typing

from asgiref.sync import async_to_sync
from yql.client.parameter_value_builder import YqlParameterValueBuilder

from crm.agency_cabinet.common.celery.utils import extract_shard_names
from crm.agency_cabinet.common.yql.base import YqlModelLoader
from crm.agency_cabinet.common.yt.base import MethodExtractor
from crm.agency_cabinet.common.db.models import BaseModel

LOGGER = logging.getLogger('celery.load_client_info')
BASE_SOURCE_PATH = '//home/direct/mysql-sync/current'

YQL_QUERY = '''
USE hahn;

DECLARE $base_source_path AS String;

SELECT
    clients.ClientID AS id,
    MAX(agency_client_relations.agency_client_id) AS agency_id,
    MAX(clients.name) AS name,
    MAX(users.FIO) AS FIO,
    COALESCE(MAX(users.login), '') AS login
FROM LIKE(
    $base_source_path,
    `clients`
) AS clients
LEFT JOIN LIKE(
    $base_source_path,
    `agency_client_relations`
) AS agency_client_relations
    ON clients.ClientID = agency_client_relations.client_client_id
LEFT JOIN LIKE(
    $base_source_path,
    `users`
) AS users
    ON clients.chief_uid = users.uid
WHERE agency_client_relations.agency_client_id IS NOT NULL
GROUP BY clients.ClientID;
'''


class ClientInfoLoader(YqlModelLoader):
    def _extract_name(self, yt_row) -> str:
        name = self._get_column_value(yt_row, 'name')
        if name is None:
            name = self._get_column_value(yt_row, 'FIO')

        return name

    def _find_duplicate(self, yt_row) -> typing.Optional[BaseModel]:
        @async_to_sync
        async def _get_client(client_id):
            async with self.db_bind:
                return await self.model.query.where(self.model.id == client_id).gino.first()

        return _get_client(self._get_column_value(yt_row, 'id'))

    def _process_duplicate(self, yt_row, db_row: BaseModel):
        @async_to_sync
        async def _update_client(client: BaseModel, client_id: int, agency_id: int, name: str, login: str):
            async with self.db_bind:
                return await client.update(
                    id=client_id,
                    agency_id=agency_id,
                    name=name,
                    login=login,
                ).apply()

        _update_client(
            db_row,
            self._get_column_value(yt_row, 'id'),
            self._get_column_value(yt_row, 'agency_id'),
            self._extract_name(yt_row),
            self._get_column_value(yt_row, 'login'),
        )


def load_single_shard_clients(
    model: typing.Type[BaseModel],
    yql_token: str,
    table_path: str,
    client_config: dict[str],
    force_load: bool,
):
    parameters = {
        '$base_source_path': YqlParameterValueBuilder.make_string(
            table_path
        ),
    }

    client_loader = ClientInfoLoader(
        table_path=table_path,
        model=model,
        columns_mapper={
            'id': 'id',
            'agency_id': 'agency_id',
            'name': MethodExtractor('_extract_name'),
            'login': 'login',
        },
        default_columns={},
        client_config=client_config,
        yql_query=YQL_QUERY,
        yql_token=yql_token,
        yql_parameters=parameters,
        force_load=force_load,
    )
    client_loader.load()


def load_client_info(model: typing.Type[BaseModel], yt_token: str, yql_token: str, force_load: bool):
    client_config = {
        'cluster': 'hahn',
        'token': yt_token,
        'config': {}
    }

    for shard in extract_shard_names(base_path=BASE_SOURCE_PATH, client_config=client_config):
        table_path = f'{shard}/straight'

        load_single_shard_clients(
            model=model,
            yql_token=yql_token,
            table_path=table_path,
            client_config=client_config,
            force_load=force_load,
        )
