import typing
from celery.beat import Scheduler, PersistentScheduler
from ylock.backends.yt import Manager as YTManager
from ylock.backends.zookeeper import Manager as ZookeeperManager
from crm.agency_cabinet.common.lock import YTLock, ZookeeperLock


class LockedScheduler(Scheduler):
    #: Maximum time to sleep between re-checking the schedule.
    max_interval = 30
    #: How often to sync the schedule (3 minutes by default)
    sync_every = 3 * 60
    #: How many seconds to wait until next try to acquiring the lock
    next_try_timeout = 30
    lock_key = 'celerybeat-lock'
    make_blocking_call = False
    blocking_timeout = None
    #: For YTLock - transaction will be aborted after %lock_timeout seconds without pings
    lock_timeout = 15
    manager = None

    @classmethod
    def configure(
            cls,
            manager: typing.Union[YTManager, ZookeeperManager],
            lock_key: str = None,
            next_try_timeout: int = None,
            blocking_timeout: int = None,
            lock_timeout: int = None,
            max_interval: int = None,
            sync_every: int = None,
            make_blocking_call: bool = None):
        cls.manager = manager
        if lock_key is not None:
            cls.lock_key = lock_key
        if next_try_timeout is not None:
            cls.next_try_timeout = next_try_timeout
        if blocking_timeout is not None:
            cls.blocking_timeout = next_try_timeout
        if max_interval is not None:
            cls.max_interval = max_interval
        if sync_every is not None:
            cls.sync_every = sync_every
        if lock_timeout is not None:
            cls.lock_timeout = lock_timeout
        if make_blocking_call is not None:
            cls.make_blocking_call = None

    def __init__(self, app, schedule=None, max_interval=None,
                 Producer=None, lazy=False, sync_every_tasks=None, **kwargs):
        super(LockedScheduler, self).__init__(
            app, schedule=schedule, max_interval=max_interval, Producer=Producer, lazy=lazy,
            sync_every_tasks=sync_every_tasks, **kwargs
        )

        if not lazy:
            self._lock: typing.Union[YTLock, ZookeeperLock] = self.manager.lock(
                self.lock_key,
                block=self.make_blocking_call,
                blocking_timeout=self.blocking_timeout,
                timeout=self.lock_timeout)
        self._running = None

    def tick(self, *args, **kwrags):
        try:
            # check if we still have lock or try to acquire
            if (self._running and self._lock.is_acquired()) or self._lock.acquire():
                if not self._running:
                    self.logger.info(
                        'Lock was acquired: %s, running ticks with interval %s',
                        self._lock.lock_id,
                        self.max_interval)
                self._running = True
                self.logger.debug('Acquired lock %s, running TICK', self._lock.lock_id)
                return super(LockedScheduler, self).tick(*args, **kwrags)
            else:
                if self._running is not False:
                    self.logger.info('Lock was not acquired, waiting for lock')
                self._running = False
                self.logger.debug('Lock was not acquired, no TICK')
                return self.next_try_timeout
        except Exception as ex:
            self.logger.warning('Error occurred during tick: %s', ex, exc_info=True)
            if self._running is True:
                self.logger.warning('Try to release lock: %s', self._lock.lock_id)
                self._lock.release()
                self._running = False
            return self.next_try_timeout


class LockedPersistentScheduler(LockedScheduler, PersistentScheduler):
    """
    Scheduler that checks lock in external locking backend for every tick
    And uses PersistentScheduler for storing the schedule
    """
