import datetime
import requests
import yt.logger as logger
import yt.wrapper as yt
from yt.wrapper import errors
from yt.clickhouse.execute import get_alias_from_env_or_raise, \
    YtClient, get_config, YtError, require, FORMAT_CLAUSE_REGEX, get_heavy_proxy_provider, get_version, \
    dumps, get_started_by_short, generate_uuid, get_option, TokenAuth, get_token, raise_for_token, \
    json, format_logging_params


def make_request(
        query: str,
        alias: str = None,
        raw: bool = None,
        format: str = None,
        settings: dict = None,
        client: yt.YtClient = None):
    """
        Mostly paste from yt.clickhouse.execute but returns response object
    """
    settings = settings or {}

    alias = alias or get_alias_from_env_or_raise()

    if client is None:
        client = YtClient(config=get_config(client=None))

    if format is not None and raw:
        raise YtError("Raw cannot be specified simultaneously with format")

    if format is not None:
        raw = True

    if raw is None:
        raw = get_config(client)["default_value_of_raw_option"]
        if not raw:
            format = "JSONEachRow"
            settings["output_format_json_quote_64bit_integers"] = False

    require(alias.startswith("*"), lambda: YtError("Operation alias should start with '*' character"))

    if FORMAT_CLAUSE_REGEX.search(query.upper()):
        raise YtError("Do not specify FORMAT clause in query; use format keyword instead")

    proxy_provder = get_heavy_proxy_provider(client)
    url = "http://{}/query".format(proxy_provder())

    params = settings

    params.update({
        "database": alias,
        "default_format": format
    })

    user_agent = "Python ClickhouseYt client " + get_version()

    headers = {
        "User-Agent": user_agent,
        "X-Started-By": dumps(get_started_by_short()),
        "X-YT-Correlation-Id": generate_uuid(get_option("_random_generator", client))
    }

    auth = TokenAuth(get_token(client=client))

    random_generator = get_option("_random_generator", client)
    request_id = "%08x" % random_generator.randrange(16**8)
    request_info = {"headers": headers, "url": url, "params": params}

    start_time = datetime.datetime.now()

    response = requests.post(
        url,
        data=query.encode('utf-8'),
        params=params,
        headers=headers,
        auth=auth,
        timeout=get_config(client)["proxy"]["request_timeout"],
        stream=True)

    if response.status_code == 401:
        raise_for_token(response, request_info)
    elif response.status_code != 200:
        if "X-Yt-Error" in response.headers:
            # This case corresponds to situation when error is initiated by out proxy code.
            error = errors.YtHttpResponseError(YtError(**json.loads(response.headers["X-Yt-Error"])))
        else:
            # This case corresponds to situation when error is forwarded from ClickHouse.
            error = errors.YtHttpResponseError(YtError("ClickHouse error: " + response.text.strip(), attributes={
                "trace_id": response.headers.get("X-YT-Trace-Id"),
                "span_id": response.headers.get("X-YT-Span-Id"),
            }))
        raise error
    finish_time = datetime.datetime.now()

    logging_params = {
        "request_id": request_id,
        "total_time": finish_time - start_time
    }
    logger.debug("Finished request execution (%s)", format_logging_params(logging_params))
    return response
