import datetime
from datetime import timezone
from crm.agency_cabinet.common.enum import BaseEnum

from dateutil.relativedelta import relativedelta


BASE_PREDICTS_SOURCE_PATH = '//home/search-research/ga/agency_rewards/forecast/v2/predict/amt/hybrid/comb_rewards/by_month'
BASE_ACTUAL_DATA_SOURCE_PATH = '//home/search-research/ga/agency_rewards/forecast/v2/predict/amt/hybrid/comb_rewards/by_month'
EARLY_PAYMENT_DATA_SOURCE_PATH = '//home/balance/prod/bo/v_ar_rewards'


class RewardsTypes(BaseEnum):
    month = 'month'
    quarter = 'quarter'
    semiyear = 'semiyear'


REWARD_TO_DURATION = {
    RewardsTypes.month: 1,
    RewardsTypes.quarter: 3,
    RewardsTypes.semiyear: 6,
}


def reward_to_duration(reward_type: RewardsTypes) -> int:
    return REWARD_TO_DURATION[reward_type]


'''
maps month to the nearest quarter start month:
March, April, May -> March
June, July, August -> June
September, October, November -> September
December, January, February -> December
'''
NEAREST_QUARTER_PERIOD_START = {
    3: 3,
    4: 3,
    5: 3,

    6: 6,
    7: 6,
    8: 6,

    9: 9,
    10: 9,
    11: 9,

    12: 12,
    1: 12,
    2: 12,
}

'''
maps month to the nearest semiyear start month:
March, April, May, June, July, August -> March
September, October, November, December, January, February -> September
'''
NEAREST_SEMIYEAR_PERIOD_START = {
    3: 3,
    4: 3,
    5: 3,
    6: 3,
    7: 3,
    8: 3,

    9: 9,
    10: 9,
    11: 9,
    12: 9,
    1: 9,
    2: 9,
}


def nearest_period_start_month(reward_type: RewardsTypes, month: int) -> int:
    if reward_type == RewardsTypes.month:
        return month
    elif reward_type == RewardsTypes.quarter:
        return NEAREST_QUARTER_PERIOD_START[month]
    elif reward_type == RewardsTypes.semiyear:
        return NEAREST_SEMIYEAR_PERIOD_START[month]


def compute_reward_type(period_from: datetime.datetime, period_to: datetime.datetime) -> str:
    period_len = relativedelta(period_to, period_from).months
    if period_len <= 1:
        return RewardsTypes.month.value
    elif period_len <= 3:
        return RewardsTypes.quarter.value
    elif period_len <= 6:
        return RewardsTypes.semiyear.value
    raise ValueError('Unknown period')


def get_start_of_current_fin_year(date: datetime.date) -> datetime.date:
    start_year = date.year
    if date.month < 3:
        start_year -= 1
    return datetime.date(year=start_year, month=3, day=1)


def get_end_of_current_fin_year(date: datetime.date) -> datetime.date:
    end_year = date.year
    if date.month >= 3:
        end_year += 1
    return datetime.date(year=end_year, month=3, day=1)


def get_start_of_current_fin_year_with_dt() -> datetime.date:
    start_fin_year = get_start_of_current_fin_year(datetime.datetime.now().date())
    return datetime.datetime(start_fin_year.year, start_fin_year.month, start_fin_year.day, tzinfo=timezone.utc)
