import os
import json
import ylock
from functools import cache
from datetime import datetime
from ylock.backends.zookeeper import LockImplWithTTL, KazooLockWrapper
from .yt import YTLock
from .zookeeper import ZookeeperLock


@cache
def monkey_patch():
    setattr(ylock.base.BaseLock, 'is_acquired', lambda self: self.check_acquired())

    def get_yt_lock(self, name, timeout=None, block=None, block_timeout=None, common_lock_name=False, **kwargs):

        # to replace slash because it is reserved symbol in path value
        name = name.replace('/', '__')

        if isinstance(name, bytes):
            try:
                name = name.decode('UTF-8')
            except UnicodeError:
                pass

        full_name = self.get_full_name(name, '/')
        if not full_name.startswith('//'):
            full_name = '//{}'.format(full_name)
        return YTLock(
            manager=self,
            name=full_name,
            timeout=timeout,
            block=block,
            block_timeout=block_timeout,
            common_lock_name=common_lock_name,
        )

    def get_zookeeper_lock(self, name, timeout=None, block=None, block_timeout=None, **kwargs):
        name = self.get_full_name(name, '/')

        if timeout is None:
            timeout = self.default_timeout
        if block_timeout is None:
            block_timeout = self.default_block_timeout

        has_lifetime = not block and timeout > 0

        if has_lifetime:
            lock = LockImplWithTTL(name, self, timeout)
        else:
            client_data = json.dumps({
                'created': datetime.now().isoformat(),
                'host': os.uname()[1],
                'pid': os.getpid(),
                'ppid': os.getppid(),
            })
            lock = KazooLockWrapper(
                self.client, name, client_data,
                block_timeout=block_timeout
            )

        self._lock_count += 1

        return ZookeeperLock(
            self, name, timeout, block, block_timeout, impl=lock
        )

    ylock.backends.yt.Manager.lock = get_yt_lock
    ylock.backends.zookeeper.Manager.lock = get_zookeeper_lock


monkey_patch()
