import socket
import time
import typing
from multiprocessing import current_process
from ylock.backends.yt import YTLock as BaseYTLock, YTTimeout, logger, DEFAULT_BLOCK_TIMEOUT, Manager as YTManager


class YTLock(BaseYTLock):

    def __init__(
            self,
            manager: YTManager,
            name: str,
            timeout: int,
            block: bool,
            block_timeout: int,
            common_lock_name: bool):
        super().__init__(manager, name, timeout, block, block_timeout, common_lock_name)
        self.lock_id = None

    def acquire(self, timeout: typing.Optional[int] = None) -> bool:
        """Block until lock is acquired, but no longer than `timeout` seconds. Returns True if lock is acquired."""

        logger.debug('will acquire %s', self._path)
        FQDN = socket.getfqdn()

        title = 'Lock tx  @ {}/{}'.format(FQDN, current_process().pid)

        transaction_kwargs = {'attributes': {'title': title}}

        if self.transaction_timeout:
            transaction_kwargs['timeout'] = self.transaction_timeout * 1000  # in msec

        self._tx = self._yt.Transaction(**transaction_kwargs)

        logger.debug('tid %s', self._tx.transaction_id)

        self._prepare()

        self._tx.__enter__()

        if self._timeout is not None:
            _timeout = self._timeout
        else:
            _timeout = timeout

        if _timeout is None:
            _timeout = DEFAULT_BLOCK_TIMEOUT

        if not self.block:
            _timeout = 0

        deadline = time.time() + _timeout

        try:
            self.lock_id = self._acquire(deadline)
        except YTTimeout:
            self.release()
            return False
        except Exception:
            self.release()
            raise

        logger.debug('acquired lock_id %s', self.lock_id)
        return True

    def release(self, remove_node: bool = True) -> bool:
        self.lock_id = None
        return super().release(remove_node)

    def is_acquired(self) -> bool:
        # check if lock still exists and this instance acquired it
        if self.lock_id is None:
            return False
        if self._yt.exists('#%s' % self.lock_id) and self._yt.get('#%s/@state' % self.lock_id) == 'acquired':
            return True

        self.lock_id = None
        return False
