import time
import asyncio
import functools
from aiohttp import web
from library.python.monlib.encoder import dumps

CONTENT_TYPE_SPACK = 'application/x-solomon-spack'
CONTENT_TYPE_JSON = 'application/json'


async def setup_metrics_registry(app: web.Application, metric_registry):
    app['metric_registry'] = metric_registry
    app['start_time'] = time.time()


class BaseMetricsHandler:

    def __init__(self, *args, **kwargs):
        pass

    async def build_metrics(self, registry, request):
        start_time = request.app['start_time']
        registry.gauge({'sensor': 'server.uptime_seconds'}).set(time.time() - start_time)

    async def __call__(self, request):
        registry = request.app['metric_registry']
        await self.build_metrics(request=request, registry=registry)
        if request.headers.get('accept', None) == CONTENT_TYPE_SPACK:
            return web.Response(body=dumps(registry), content_type=CONTENT_TYPE_SPACK)

        return web.Response(body=dumps(registry, format='json'), content_type=CONTENT_TYPE_JSON)


async def run_ping_service(metric_registry, metrics_handler_cls=BaseMetricsHandler,
                           metrics_handler_kwargs=None,
                           metric_handler_args=()):
    if metrics_handler_kwargs is None:
        metrics_handler_kwargs = {}

    async def ping(request):
        return web.Response(text='pong')

    app = web.Application()
    setup_metrics_registry_bounded = functools.partial(setup_metrics_registry, metric_registry=metric_registry)
    app.on_startup.append(setup_metrics_registry_bounded)

    app.add_routes([web.get('/ping', ping)])
    app.add_routes([web.get('/metrics', metrics_handler_cls(*metric_handler_args, **metrics_handler_kwargs))])
    runner = web.AppRunner(app, handle_signals=True, access_log_class=web.AccessLogger,
                           access_log_format=web.AccessLogger.LOG_FORMAT, )
    try:
        await runner.setup()
        site = web.TCPSite(runner, 'localhost', 1337)
        await site.start()
        while True:
            await asyncio.sleep(3600)
    finally:
        await runner.cleanup()
