import typing
import json
from webargs import aiohttpparser
from aiohttp.web_request import Request
from aiohttp import web
from aiohttp_apispec import AiohttpApiSpec


class AIOHTTPParserEnhanced(aiohttpparser.AIOHTTPParser):
    def handle_invalid_json_error(
        self,
        error: typing.Union[json.JSONDecodeError, UnicodeDecodeError],
        req: Request,
        *args,
        **kwargs
    ) -> typing.NoReturn:
        error_class = aiohttpparser.exception_map[400]
        message = "Invalid JSON body."
        raise error_class(text=message)


parser = AIOHTTPParserEnhanced()


def enum_to_properties(self, field, **kwargs):
    import marshmallow_enum
    if isinstance(field, marshmallow_enum.EnumField):
        return {'type': 'string', 'enum': [m.name for m in field.enum]}
    return {}


class AiohttpApiSpecEnhanced(AiohttpApiSpec):

    def register(self, app: web.Application, in_place: bool = False):
        if self._registered is True:
            return None

        app["_apispec_request_data_name"] = self._request_data_name

        app["_apispec_parser"] = parser
        if self.error_callback:
            app["_apispec_parser"].error_callback = self.error_callback

        if in_place:
            self._register(app)
        else:

            async def doc_routes(app_):
                self._register(app_)

            app.on_startup.append(doc_routes)

        self._registered = True

        if self.url is not None:
            async def swagger_handler(request):
                return web.json_response(request.app["swagger_dict"])

            app.router.add_route("GET", self.url, swagger_handler, name="swagger.spec")

            if self.swagger_path is not None:
                self._add_swagger_web_page(app, self.static_path, self.swagger_path)


def setup_aiohttp_apispec(
    app: web.Application,
    *,
    title: str = "API documentation",
    version: str = "0.0.1",
    url: typing.Optional[str] = "/api/docs/swagger.json",
    request_data_name: str = "data",
    swagger_path: str = None,
    static_path: str = '/static/swagger',
    error_callback=None,
    in_place: bool = False,
    prefix: str = '',
    **kwargs
) -> None:
    spec = AiohttpApiSpecEnhanced(
        url=url,
        app=app,
        request_data_name=request_data_name,
        title=title,
        version=version,
        swagger_path=swagger_path,
        static_path=static_path,
        error_callback=error_callback,
        in_place=in_place,
        prefix=prefix,
        **kwargs
    )
    spec.plugin.converter.add_attribute_function(enum_to_properties)


def setup_aiohttp_subapp_apispec(
    app,
    subapp
):
    # для работы ValidateMiddleware
    subapp['_apispec_parser'] = app['_apispec_parser']
    subapp['_apispec_request_data_name'] = app['_apispec_request_data_name']
