import typing
import time
from dataclasses import dataclass
from library.python.monlib.metric_registry import MetricRegistry, HistogramType

from aiohttp.web_exceptions import HTTPException
from aiohttp.web_middlewares import middleware
from aiohttp.web_request import Request
from aiohttp.web_response import Response


@dataclass
@middleware
class RateMonitoringMiddleware:
    async def __call__(self, request: Request, handler: typing.Callable[[Request], typing.Awaitable[Response]]) -> Response:
        start_time = time.time()
        endpoint = request.match_info.route.resource.canonical if request.match_info.route.resource else 'unknown_path'
        status_code = 500
        try:
            response = await handler(request)
            status_code = response.status
            return response
        except HTTPException as e:
            status_code = e.status
            raise
        finally:
            registry: MetricRegistry = request.app['metric_registry']

            if 200 <= status_code < 300:
                registry.rate({'path': endpoint, 'sensor': 'http.ok'}).inc()
            elif 400 <= status_code < 500:
                registry.rate({'path': endpoint, 'sensor': 'http.client_error'}).inc()
            else:
                registry.rate({'path': endpoint, 'sensor': 'http.server_error'}).inc()

            if 100 <= status_code < 600:
                registry.rate({'path': endpoint, 'sensor': 'http.requests', 'status': str(status_code)}).inc()
            else:
                registry.rate({'path': endpoint, 'sensor': 'http.unknown_status', 'status': str(status_code)}).inc()

            registry.histogram_rate(
                {'path': endpoint, 'sensor': 'server.response_time_milliseconds'},
                HistogramType.Explicit, buckets=[1, 10, 20, 50, 200, 500, 1000, 2000, 5000, 10000]).collect((time.time() - start_time) * 1000)
