import typing
from dataclasses import dataclass

from aiohttp.web_exceptions import HTTPException
from aiohttp.web_middlewares import middleware
from aiohttp.web_request import Request
from aiohttp.web_response import Response

from crm.agency_cabinet.common.request_id_utils import generate_request_id, REQUEST_ID_HEADER_NAME, REQUEST_ID_VAR, \
    set_request_id_header


@dataclass
@middleware
class RequestIdMiddleware:
    request_id_factory: typing.Callable[[], str] = generate_request_id

    async def __call__(self, request: Request, handler: typing.Callable[[Request], typing.Awaitable[Response]]) -> Response:
        request_id = request.headers.get(REQUEST_ID_HEADER_NAME) or self.request_id_factory()
        token = REQUEST_ID_VAR.set(request_id)
        request['request_id'] = request_id
        try:
            response = await handler(request)
            set_request_id_header(response)
            return response
        except HTTPException as e:
            set_request_id_header(e)
            raise
        finally:
            REQUEST_ID_VAR.reset(token)
