# -*- coding: utf-8 -*-
from contextlib import AbstractAsyncContextManager
from types import TracebackType
from typing import Any, Optional, Type

from smb.common.rmq.rpc.client import RmqRpcClient
from smb.common.rmq.rpc.server import RmqRpcServer

from crm.agency_cabinet.agencies.client import AgenciesClient
from crm.agency_cabinet.certificates.client import Client as CertificatesClient
from crm.agency_cabinet.client_bonuses.client import Client as ClientBonusesClient
from crm.agency_cabinet.grants.client import GrantsClient
from crm.agency_cabinet.rewards.client import RewardsClient
from crm.agency_cabinet.documents.client import DocumentsClient
from crm.agency_cabinet.ord.client import OrdClient


class ServiceDiscovery(AbstractAsyncContextManager):
    __rpc_client: Optional[RmqRpcClient]
    __with_rpc_client: bool

    agencies: AgenciesClient
    certificates: CertificatesClient
    client_bonuses: ClientBonusesClient
    grants: GrantsClient
    rewards: RewardsClient
    documents: DocumentsClient
    ord: OrdClient

    def __init__(self, ampq_url: str, with_rpc_client: bool = True):
        self.__with_rpc_client = with_rpc_client
        self.__rpc_client = None
        self.__ampq_url = ampq_url

    @property
    def amqp_url(self) -> str:
        return self.__ampq_url

    @property
    def rpc_client(self) -> RmqRpcClient:
        if not self.__with_rpc_client:
            raise RuntimeError('Service discovery configured without rpc client!')
        return self.__rpc_client

    def rpc_server(self, queue_name: str, **kwargs: Any) -> RmqRpcServer:
        return RmqRpcServer(amqp_url=self.amqp_url, queue_name=queue_name, **kwargs)

    async def connect(self) -> None:
        if self.__with_rpc_client:
            self.__rpc_client = RmqRpcClient(amqp_url=self.amqp_url)
            await self.rpc_client.connect()

            self._attach_service_rpc_clients()

    async def __aenter__(self) -> 'ServiceDiscovery':
        await self.connect()
        return self

    async def __aexit__(self, exc_type: Optional[Type[BaseException]], exc_value: Optional[BaseException],
                        traceback: Optional[TracebackType]):
        if self.__with_rpc_client:
            await self.rpc_client.close()

    def _attach_service_rpc_clients(self):
        self.grants = GrantsClient(self.rpc_client)
        self.certificates = CertificatesClient(self.rpc_client)
        self.client_bonuses = ClientBonusesClient(self.rpc_client)
        self.agencies = AgenciesClient(self.rpc_client)
        self.rewards = RewardsClient(self.rpc_client)
        self.documents = DocumentsClient(self.rpc_client)
        self.ord = OrdClient(self.rpc_client)
