import datetime
import typing
import asyncio
import logging

import aiohttp
from urllib import parse
from io import BytesIO
from crm.agency_cabinet.common.enum import BaseEnum
from crm.agency_cabinet.common.server.common.tvm import TvmClient
from crm.agency_cabinet.common.consts.tvm import TVMIdTest


LOGGER = logging.getLogger('yadoc.client')


class YaDocType(BaseEnum):
    act = 'ACT'
    invoice = 'BILL'  # счёт
    facture = 'INV'  # счёт-фактура


class YaDocClient:

    def __init__(
        self,
        endpoint_url: str,
        tvm_client: TvmClient,
        yadoc_tvm_id: int = TVMIdTest.yadoc.value,
        *args, **kwargs
    ):
        self.url = endpoint_url
        self.session = aiohttp.ClientSession(*args, **kwargs)
        self.tvm_client = tvm_client
        self.yadoc_tvm_id = yadoc_tvm_id

    async def _get_ticket(self):
        ticket = await self.tvm_client.get_service_ticket(destination=str(self.yadoc_tvm_id))
        return ticket

    def _get_request_context(self, method, *args, **kwargs):
        return getattr(self.session, method)(*args, **kwargs)

    async def _make_request(self, api_path, method='get', data=None, headers=None, params=None, json=None):
        if headers is None:
            headers = {}
        url = parse.urljoin(self.url, api_path)
        headers.update({'X-Ya-Service-Ticket': await self._get_ticket()})
        return self._get_request_context(method, url=url, headers=headers, data=data, params=params, json=json)

    async def get_doc_url(self, yadoc_id: int) -> str:
        api_path = f'/public/api/documents/{yadoc_id}/url'

        async with await self._make_request(api_path) as r:
            response = await r.json()
            return response.get('url')

    async def get_doc(self, yadoc_id: int) -> BytesIO:
        api_path = f'/public/api/documents/{yadoc_id}/download'

        async with await self._make_request(api_path) as r:
            binary_content = BytesIO(await r.read())
            return binary_content

    async def _def_fetch_get_docs_info_pages(self, api_path, request_data, params):
        LOGGER.info('Process page № %s', params['page'])
        async with await self._make_request(api_path, json=request_data, params=params, method='post') as r:
            return await r.json()

    async def get_docs_info(
        self,
        date_from: datetime.date,
        date_to: datetime.date,
        contract_ids: typing.List[int] = None,
        doc_types: typing.List[str] = None,
        bill_numbers: typing.List[str] = None,
        doc_numbers: typing.List[str] = None,
        exclude_reversed: bool = True,
        page_size: int = 20
    ) -> typing.AsyncIterator[typing.Dict]:
        api_path = '/public/api/v1/documents'
        request_data = {
            'date_from': date_from.strftime('%Y-%m-%d'),
            'date_to': date_to.strftime('%Y-%m-%d'),
            'exclude_reversed': exclude_reversed
        }
        if contract_ids:
            request_data['contract_id'] = contract_ids

        if doc_types:
            request_data['doc_type'] = doc_types

        if bill_numbers:
            request_data['bill_number'] = bill_numbers

        if doc_numbers:
            request_data['doc_number'] = doc_numbers

        current_page = 0

        params = {
            'page': current_page,
            'size': page_size
        }

        async with await self._make_request(
            api_path,
            json=request_data,
            params=params,
            method='post'
        ) as r:
            response = await r.json()
            total_elements = response.get('totalElements', 0)
            LOGGER.info('Total elements: %s', total_elements)
            elements_left = total_elements
            content = response.get('content') or []
            for data in content:
                elements_left -= 1
                LOGGER.debug('Elements left: %s', elements_left)
                yield data

        if elements_left > 0:
            pages_left = (elements_left - 1)//page_size + 1
            last_page = 1
            step = 10
            while pages_left > 0:
                left_b = last_page
                right_b = left_b + min(step, pages_left)
                pages_left -= step
                futures = asyncio.as_completed(
                    [self._def_fetch_get_docs_info_pages(
                        api_path,
                        request_data,
                        params={
                            'page': i,
                            'size': page_size,
                        }
                    ) for i in range(left_b, right_b)],
                )
                last_page = right_b
                for future in futures:
                    # retry on exception???
                    res = await future
                    content = res.get('content') or []
                    for data in content:
                        elements_left -= 1
                        LOGGER.debug('Elements left: %s', elements_left)
                        yield data

                await asyncio.sleep(0.5)

    async def get_first_doc_info(
        self,
        date_from: datetime.date,
        date_to: datetime.date,
        contract_ids: typing.List[int] = None,
        doc_types: typing.List[str] = None,
        bill_numbers: typing.List[str] = None,
        doc_numbers: typing.List[str] = None,
        exclude_reversed: bool = True,
    ) -> typing.Dict:
        docs_info = self.get_docs_info(
            date_from,
            date_to,
            contract_ids,
            doc_types,
            bill_numbers,
            doc_numbers,
            exclude_reversed
        )
        async for doc_info in docs_info:
            return doc_info

    async def close_session(self):
        await self.session.close()

    async def init_session(self, *args, **kwargs):
        self.session = aiohttp.ClientSession(*args, **kwargs)
