import datetime
import logging

from abc import ABC

from sqlalchemy import and_
from yt.wrapper import YtClient

from crm.agency_cabinet.common.db.models import YtSync


LOGGER = logging.getLogger('yt.loaders')


class BaseLoader(ABC):
    table_update_attr = 'modification_time'

    def __init__(self, yt_client: YtClient):
        self._yt_client = yt_client
        self._updated_at = None

    def _get_ytsync_tag(self) -> str:
        return self.__class__.__name__

    def _get_ytsync_path(self) -> str:
        pass

    @staticmethod
    def _prepare_table_path(table_path):
        return f'//{table_path.strip("/")}/'

    async def _is_should_execute(self):
        ytsync_path = self._get_ytsync_path()
        row: YtSync = await YtSync.query.where(
            and_(YtSync.path == ytsync_path, YtSync.tag == self._get_ytsync_tag())
        ).gino.first()

        # there is no record with provided table path, so we should add it
        if row is None:
            return True
        elif not row.enabled:
            return False

        LOGGER.debug(f'Make check for {ytsync_path} with last_updated_at: {self._updated_at} vs {row.last_used_update}')
        return row.last_used_update != self._updated_at

    def _retrieve_tables_updated_at(self):
        pass

    async def _load(self, *args, **kwargs) -> bool:
        pass

    async def _after_load(self):
        ytsync_path = self._get_ytsync_path()
        row: YtSync = await YtSync.query.where(
            and_(YtSync.path == ytsync_path, YtSync.tag == self._get_ytsync_tag())
        ).gino.first()

        if row is None:
            await YtSync.create(path=ytsync_path, tag=self._get_ytsync_tag(), last_used_update=self._updated_at)
        else:
            await row.update(
                last_used_update=self._updated_at,
                updated_at=datetime.datetime.now(tz=datetime.timezone.utc)
            ).apply()

    async def load(self, force_load: bool = False, *args, **kwargs) -> bool:
        self._updated_at = self._retrieve_tables_updated_at()
        if not force_load and not await self._is_should_execute():
            return False

        if not await self._load(*args, **kwargs):
            return False

        await self._after_load()
        return True
