from datetime import datetime

from crm.agency_cabinet.common.server.common.structs.common import UrlResponse
from crm.agency_cabinet.common.client import BaseClient
from crm.agency_cabinet.common.client.exceptions import ProtocolError
from crm.agency_cabinet.documents.proto import common_pb2, request_pb2, contracts_pb2, invoices_pb2, payments_pb2, \
    acts_pb2, agreements_pb2
from crm.agency_cabinet.documents.common import QUEUE
from crm.agency_cabinet.documents.common import structs

from crm.agency_cabinet.documents.client.exceptions import NoSuchContractException, \
    UnsuitableAgencyException, NoSuchInvoiceException, \
    NoSuchDocumentException, FileNotFoundException, NoSuchActException, NoSuchFactureException, NoSuchAgreementException


__all__ = [
    'NoSuchContractException',
    'UnsuitableAgencyException',
    'NoSuchInvoiceException',
    'NoSuchDocumentException',
    'FileNotFoundException',
    'DocumentsClient',
    'NoSuchAgreementException'
]


class DocumentsClient(BaseClient):
    queue = QUEUE

    async def ping(self) -> str:
        _, data = await self._send_message(
            request_pb2.RpcRequest(ping=common_pb2.Empty()),
            common_pb2.PingOutput
        )
        return str(data)

    async def list_contracts(self, agency_id: int) -> list[structs.Contract]:
        request = request_pb2.RpcRequest(
            list_contracts=structs.ListContractsInput(
                agency_id=agency_id
            ).to_proto()
        )

        result, data = await self._send_message(
            request, contracts_pb2.ListContractsOutput
        )
        if result == "contracts":
            return [structs.Contract.from_proto(contract) for contract in data.contracts]

        raise ProtocolError("Unexpected response")

    async def get_contract_info(self, agency_id: int, contract_id: int) -> structs.Contract:
        request = request_pb2.RpcRequest(
            get_contract_info=contracts_pb2.GetContractInfoInput(
                agency_id=agency_id,
                contract_id=contract_id
            )
        )
        result, data = await self._send_message(
            request,
            contracts_pb2.GetContractInfoOutput
        )

        if result == 'contract':
            return structs.GetContractInfoOutput.from_proto(data).contract
        elif result == 'no_such_contract':
            raise NoSuchContractException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()

        raise ProtocolError("Unexpected response")

    async def list_invoices(self,
                            agency_id: int,
                            contract_id: int = None,
                            date_from: datetime = None,
                            date_to: datetime = None,
                            status: structs.InvoiceStatus = None,
                            limit: int = None,
                            offset: int = None,
                            search_query: str = None,
                            ) -> list[structs.Invoice]:
        request = request_pb2.RpcRequest(
            list_invoices=structs.ListInvoicesInput(
                agency_id=agency_id,
                contract_id=contract_id,
                date_from=date_from,
                date_to=date_to,
                status=status,
                limit=limit,
                offset=offset,
                search_query=search_query
            ).to_proto()
        )

        result, data = await self._send_message(
            request, invoices_pb2.ListInvoicesOutput
        )
        if result == "invoices":
            return [structs.Invoice.from_proto(invoice) for invoice in data.invoices]

        raise ProtocolError("Unexpected response")

    async def list_payments(
        self,
        agency_id: int,
        invoice_id: int,
        contract_id: int,
        limit: int,
        offset: int,
        date_to: datetime,
        date_from: datetime,
    ) -> list[structs.Payment]:

        request = request_pb2.RpcRequest(
            list_payments=structs.ListPaymentsInput(
                agency_id=agency_id,
                contract_id=contract_id,
                invoice_id=invoice_id,
                limit=limit,
                offset=offset,
                date_from=date_from,
                date_to=date_to,
            ).to_proto()
        )

        result, data = await self._send_message(
            request, payments_pb2.ListPaymentsOutput
        )

        if result == "payments":
            return [structs.Payment.from_proto(payment) for payment in data.payments]

        raise ProtocolError("Unexpected response")

    async def list_acts(
        self,
        agency_id: int,
        invoice_id: int,
        contract_id: int,
        limit: int,
        offset: int,
        date_to: datetime,
        date_from: datetime,
        search_query: str,
    ) -> list[structs.Act]:

        request = request_pb2.RpcRequest(
            list_acts=structs.ListActsInput(
                agency_id=agency_id,
                contract_id=contract_id,
                invoice_id=invoice_id,
                limit=limit,
                offset=offset,
                date_from=date_from,
                date_to=date_to,
                search_query=search_query,
            ).to_proto()
        )

        result, data = await self._send_message(
            request, acts_pb2.ListActsOutput
        )

        if result == "acts":
            return [structs.Act.from_proto(act) for act in data.acts]

        raise ProtocolError("Unexpected response")

    async def get_invoice_info(self, agency_id: int, invoice_id: int) -> structs.DetailedInvoice:
        request = request_pb2.RpcRequest(
            get_invoice_info=invoices_pb2.GetInvoiceInfoInput(
                agency_id=agency_id,
                invoice_id=invoice_id
            )
        )
        result, data = await self._send_message(
            request,
            invoices_pb2.GetInvoiceInfoOutput
        )

        if result == 'invoice':
            return structs.DetailedInvoice.from_proto(data)
        elif result == 'no_such_invoice':
            raise NoSuchInvoiceException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()

    async def list_agreements(
        self,
        agency_id: int,
        contract_id: int,
        limit: int,
        offset: int,
        date_to: datetime,
        date_from: datetime,
        search_query: str,
    ) -> list[structs.Agreement]:
        request = request_pb2.RpcRequest(
            list_agreements=structs.ListAgreementsInput(
                agency_id=agency_id,
                contract_id=contract_id,
                limit=limit,
                offset=offset,
                date_from=date_from,
                date_to=date_to,
                search_query=search_query,
            ).to_proto()
        )

        result, data = await self._send_message(
            request, agreements_pb2.ListAgreementsOutput
        )
        if result == "agreements":
            return [structs.Agreement.from_proto(agreement) for agreement in data.agreements]

        raise ProtocolError("Unexpected response")

    async def get_invoice_url(self, agency_id: int, invoice_id: int) -> str:
        request = request_pb2.RpcRequest(
            get_invoice_url=invoices_pb2.GetInvoiceUrlInput(
                agency_id=agency_id,
                invoice_id=invoice_id
            )
        )

        result, data = await self._send_message(
            request,
            invoices_pb2.GetInvoiceUrlOutput
        )

        if result == 'result':
            return UrlResponse.from_proto(data).url
        elif result == 'no_such_invoice':
            raise NoSuchInvoiceException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()
        elif result == 'file_not_found':
            raise FileNotFoundException()
        raise ProtocolError('Unexpected response')

    async def get_act_url(self, agency_id: int, act_id: int) -> str:
        request = request_pb2.RpcRequest(
            get_act_url=acts_pb2.GetActUrlInput(
                agency_id=agency_id,
                act_id=act_id
            )
        )

        result, data = await self._send_message(
            request,
            acts_pb2.GetActUrlOutput
        )

        if result == 'result':
            return UrlResponse.from_proto(data).url
        elif result == 'no_such_act':
            raise NoSuchActException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()
        elif result == 'file_not_found':
            raise FileNotFoundException()
        raise ProtocolError('Unexpected response')

    async def get_facture_url(self, agency_id: int, facture_id: int) -> str:
        request = request_pb2.RpcRequest(
            get_facture_url=invoices_pb2.GetFactureUrlInput(
                agency_id=agency_id,
                facture_id=facture_id
            )
        )

        result, data = await self._send_message(
            request,
            invoices_pb2.GetFactureUrlOutput
        )

        if result == 'result':
            return UrlResponse.from_proto(data).url
        elif result == 'no_such_facture':
            raise NoSuchFactureException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()
        elif result == 'file_not_found':
            raise FileNotFoundException()
        raise ProtocolError('Unexpected response')

    async def get_contract_url(self, agency_id: int, contract_id: int) -> str:
        request = request_pb2.RpcRequest(
            get_contract_url=contracts_pb2.GetContractUrlInput(
                agency_id=agency_id,
                contract_id=contract_id
            )
        )

        result, data = await self._send_message(
            request,
            contracts_pb2.GetContractUrlOutput
        )

        if result == 'result':
            return UrlResponse.from_proto(data).url
        elif result == 'no_such_contract':
            raise NoSuchContractException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()
        elif result == 'file_not_found':
            raise FileNotFoundException()
        raise ProtocolError('Unexpected response')

    async def get_agreement_url(self, agency_id: int, agreement_id: int) -> str:
        request = request_pb2.RpcRequest(
            get_agreement_url=agreements_pb2.GetAgreementUrlInput(
                agency_id=agency_id,
                agreement_id=agreement_id
            )
        )

        result, data = await self._send_message(
            request,
            agreements_pb2.GetAgreementUrlOutput
        )

        if result == 'result':
            return UrlResponse.from_proto(data).url
        elif result == 'no_such_agreement':
            raise NoSuchAgreementException()
        elif result == 'unsuitable_agency':
            raise UnsuitableAgencyException()
        elif result == 'file_not_found':
            raise FileNotFoundException()
        raise ProtocolError('Unexpected response')
