from datetime import datetime
from decimal import Decimal
from typing import Optional

from dataclasses import dataclass
from smb.common.helpers import PbDatetimeConverter

from crm.agency_cabinet.common.proto_utils import decimal_to_string
from crm.agency_cabinet.documents.proto import acts_pb2


@dataclass
class Act:
    act_id: int
    eid: str
    invoice_id: int
    invoice_eid: str
    currency: str
    amount: Decimal
    contract_id: int
    contract_eid: str
    date: datetime

    @classmethod
    def from_proto(cls, message: acts_pb2.Act) -> 'Act':
        return cls(
            act_id=message.act_id,
            invoice_id=message.invoice_id,
            invoice_eid=message.invoice_eid,
            eid=message.eid,
            currency=message.currency,
            amount=Decimal(message.amount),
            contract_id=message.contract_id,
            contract_eid=message.contract_eid,
            date=PbDatetimeConverter().to_datetime(message.date)
        )

    def to_proto(self) -> acts_pb2.Act:
        return acts_pb2.Act(
            act_id=self.act_id,
            invoice_id=self.invoice_id,
            invoice_eid=self.invoice_eid,
            eid=self.eid,
            currency=self.currency,
            amount=decimal_to_string(self.amount),
            contract_id=self.contract_id,
            contract_eid=self.contract_eid,
            date=PbDatetimeConverter().from_datetime(self.date),
        )


@dataclass
class ListActsInput:
    agency_id: int
    contract_id: Optional[int] = None
    invoice_id: Optional[int] = None
    limit: Optional[int] = None
    offset: Optional[int] = None
    date_from: Optional[datetime] = None
    date_to: Optional[datetime] = None
    search_query: Optional[str] = None

    @classmethod
    def from_proto(cls, message: acts_pb2.ListActsInput) -> "ListActsInput":
        return cls(
            agency_id=message.agency_id,
            contract_id=message.contract_id if message.HasField('contract_id') else None,
            invoice_id=message.invoice_id if message.HasField('invoice_id') else None,
            limit=message.limit if message.HasField('limit') else None,
            offset=message.offset if message.HasField('offset') else None,
            date_from=PbDatetimeConverter().to_datetime(message.date_from) if message.HasField('date_from') else None,
            date_to=PbDatetimeConverter().to_datetime(message.date_to) if message.HasField('date_to') else None,
            search_query=message.search_query if message.HasField('search_query') else None,
        )

    def to_proto(self) -> acts_pb2.ListActsInput:
        return acts_pb2.ListActsInput(
            agency_id=self.agency_id,
            contract_id=self.contract_id,
            invoice_id=self.invoice_id,
            limit=self.limit,
            offset=self.offset,
            date_from=PbDatetimeConverter().from_datetime(self.date_from) if self.date_from else None,
            date_to=PbDatetimeConverter().from_datetime(self.date_to) if self.date_to else None,
            search_query=self.search_query,
        )


@dataclass
class GetActUrlInput:
    agency_id: int
    act_id: int

    @classmethod
    def from_proto(cls, message: acts_pb2.GetActUrlInput) -> 'GetActUrlInput':
        return cls(
            agency_id=message.agency_id,
            act_id=message.act_id
        )

    def to_proto(self) -> acts_pb2.GetActUrlInput:
        return acts_pb2.GetActUrlInput(
            agency_id=self.agency_id,
            act_id=self.act_id
        )
