from dataclasses import dataclass
from datetime import datetime
from typing import Optional

from smb.common.helpers import PbDatetimeConverter

from crm.agency_cabinet.documents.proto import agreements_pb2


@dataclass
class Agreement:
    agreement_id: int
    name: str
    got_scan: bool
    got_original: bool
    date: datetime

    @classmethod
    def from_proto(cls, message: agreements_pb2.Agreement) -> 'Agreement':
        return cls(
            agreement_id=message.agreement_id,
            name=message.name,
            got_scan=message.got_scan,
            got_original=message.got_original,
            date=PbDatetimeConverter().to_datetime(message.date),
        )

    def to_proto(self) -> agreements_pb2.Agreement:
        return agreements_pb2.Agreement(
            agreement_id=self.agreement_id,
            name=self.name,
            got_scan=self.got_scan,
            got_original=self.got_original,
            date=PbDatetimeConverter().from_datetime(self.date),
        )


@dataclass
class ListAgreementsInput:
    agency_id: int
    contract_id: int
    limit: Optional[int] = None
    offset: Optional[int] = None
    date_from: Optional[datetime] = None
    date_to: Optional[datetime] = None
    search_query: Optional[str] = None

    @classmethod
    def from_proto(cls, message: agreements_pb2.ListAgreementsInput) -> "ListAgreementsInput":
        return cls(
            agency_id=message.agency_id,
            contract_id=message.contract_id,
            limit=message.limit if message.HasField('limit') else None,
            offset=message.offset if message.HasField('offset') else None,
            date_from=PbDatetimeConverter().to_datetime(message.date_from) if message.HasField('date_from') else None,
            date_to=PbDatetimeConverter().to_datetime(message.date_to) if message.HasField('date_to') else None,
            search_query=message.search_query if message.HasField('search_query') else None,
        )

    def to_proto(self) -> agreements_pb2.ListAgreementsInput:
        return agreements_pb2.ListAgreementsInput(
            agency_id=self.agency_id,
            contract_id=self.contract_id,
            limit=self.limit,
            offset=self.offset,
            date_from=PbDatetimeConverter().from_datetime(self.date_from) if self.date_from else None,
            date_to=PbDatetimeConverter().from_datetime(self.date_to) if self.date_to else None,
            search_query=self.search_query,
        )


@dataclass
class GetAgreementUrlInput:
    agency_id: int
    agreement_id: int

    @classmethod
    def from_proto(cls, message: agreements_pb2.GetAgreementUrlInput) -> 'GetAgreementUrlInput':
        return cls(
            agency_id=message.agency_id,
            agreement_id=message.agreement_id
        )

    def to_proto(self) -> agreements_pb2.GetAgreementUrlInput:
        return agreements_pb2.GetAgreementUrlInput(
            agency_id=self.agency_id,
            agreement_id=self.agreement_id
        )
