from decimal import Decimal
from typing import Optional

from dataclasses import dataclass
from datetime import datetime
from crm.agency_cabinet.common.enum import BaseEnum

from smb.common.helpers import Converter, PbDatetimeConverter

from crm.agency_cabinet.common.consts import Services, PaymentType
from crm.agency_cabinet.documents.proto import contracts_pb2
from crm.agency_cabinet.common.proto_utils import decimal_to_string


class ContractStatus(BaseEnum):
    not_signed = 'not_signed'
    valid = 'valid'


contract_status_converter = Converter(
    [
        (contracts_pb2.ContractStatus.not_signed, ContractStatus.not_signed),
        (contracts_pb2.ContractStatus.valid, ContractStatus.valid),
    ]
)

contract_payment_type_converter = Converter(
    [
        (contracts_pb2.PaymentType.prepayment, PaymentType.prepayment),
        (contracts_pb2.PaymentType.postpayment, PaymentType.postpayment),
    ]
)

contract_service_converter = Converter(
    [
        (contracts_pb2.Services.media, Services.media),
        (contracts_pb2.Services.direct, Services.direct),
        (contracts_pb2.Services.zen, Services.zen),
        (contracts_pb2.Services.video, Services.video),
        (contracts_pb2.Services.sprav, Services.sprav),
        (contracts_pb2.Services.business, Services.business),
        (contracts_pb2.Services.early_payment, Services.early_payment),
    ]
)


@dataclass
class Contract:
    contract_id: int
    eid: str
    status: ContractStatus
    payment_type: Optional[PaymentType]
    services: list[Services]
    finish_date: datetime
    inn: Optional[str] = None
    credit_limit: Optional[Decimal] = None
    signing_date: Optional[datetime] = None

    @classmethod
    def from_proto(cls, message: contracts_pb2.Contract) -> 'Contract':
        signing_date = None
        if message.signing_date.seconds:
            signing_date = PbDatetimeConverter().to_datetime(message.signing_date)

        return cls(
            contract_id=message.contract_id,
            eid=message.eid,
            inn=message.inn if message.HasField('inn') else None,
            status=contract_status_converter.forward(message.status),
            payment_type=contract_payment_type_converter.forward(message.payment_type) if message.HasField('payment_type') else None,
            services=[contract_service_converter.forward(service) for service in message.services],
            signing_date=signing_date,
            finish_date=PbDatetimeConverter().to_datetime(message.finish_date),
            credit_limit=Decimal(message.credit_limit) if message.HasField('credit_limit') else None,
        )

    def to_proto(self) -> contracts_pb2.Contract:
        signing_date = None

        if self.signing_date:
            signing_date = PbDatetimeConverter().from_datetime(self.signing_date)

        return contracts_pb2.Contract(
            contract_id=self.contract_id,
            eid=self.eid,
            inn=self.inn or None,
            status=contract_status_converter.reversed(self.status),
            payment_type=contract_payment_type_converter.reversed(self.payment_type) if self.payment_type else None,
            services=[contract_service_converter.reversed(service) for service in self.services],
            signing_date=signing_date,
            finish_date=PbDatetimeConverter().from_datetime(self.finish_date),
            credit_limit=decimal_to_string(self.credit_limit) if self.credit_limit is not None else None,
        )


@dataclass
class ListContractsInput:
    agency_id: int

    @classmethod
    def from_proto(cls, message: contracts_pb2.ListContractsInput) -> 'ListContractsInput':
        return cls(agency_id=message.agency_id)

    def to_proto(self) -> contracts_pb2.ListContractsInput:
        return contracts_pb2.ListContractsInput(agency_id=self.agency_id)


@dataclass
class GetContractInfoInput:
    contract_id: int
    agency_id: int

    @classmethod
    def from_proto(cls, message: contracts_pb2.GetContractInfoInput) -> 'GetContractInfoInput':
        return cls(agency_id=message.agency_id, contract_id=message.contract_id)

    def to_proto(self) -> contracts_pb2.GetContractInfoInput:
        return contracts_pb2.GetContractInfoInput(agency_id=self.agency_id, contract_id=self.contract_id)


@dataclass
class GetContractInfoOutput:
    contract: Contract

    @classmethod
    def from_proto(cls, message: contracts_pb2.Contract) -> 'GetContractInfoOutput':
        return cls(contract=Contract.from_proto(message))

    def to_proto(self) -> contracts_pb2.Contract:
        return self.contract.to_proto()


@dataclass
class GetContractUrlInput:
    agency_id: int
    contract_id: int

    @classmethod
    def from_proto(cls, message: contracts_pb2.GetContractUrlInput) -> 'GetContractUrlInput':
        return cls(
            agency_id=message.agency_id,
            contract_id=message.contract_id
        )

    def to_proto(self) -> contracts_pb2.GetContractUrlInput:
        return contracts_pb2.GetContractUrlInput(
            agency_id=self.agency_id,
            contract_id=self.contract_id
        )
