import typing
from decimal import Decimal
from typing import Optional

from dataclasses import dataclass
from datetime import datetime
from crm.agency_cabinet.common.enum import BaseEnum

from smb.common.helpers import Converter, PbDatetimeConverter

from crm.agency_cabinet.common.proto_utils import decimal_to_string
from crm.agency_cabinet.documents.proto import invoices_pb2


class InvoiceStatus(BaseEnum):
    paid = 'paid'
    not_paid = 'not_paid'


invoice_status_converter = Converter(
    [
        (invoices_pb2.InvoiceStatus.paid, InvoiceStatus.paid),
        (invoices_pb2.InvoiceStatus.not_paid, InvoiceStatus.not_paid),
    ]
)


@dataclass
class Invoice:
    invoice_id: int
    eid: str
    contract_id: int
    contract_eid: str
    amount: Decimal
    currency: str
    status: InvoiceStatus
    date: datetime
    payment_date: Optional[datetime]
    has_facture: bool

    @classmethod
    def from_proto(cls, message: invoices_pb2.Invoice) -> 'Invoice':
        return Invoice(
            invoice_id=message.invoice_id,
            eid=message.eid,
            contract_id=message.contract_id,
            contract_eid=message.contract_eid,
            amount=Decimal(message.amount),
            currency=message.currency,
            status=invoice_status_converter.forward(message.status),
            date=PbDatetimeConverter().to_datetime(message.date),
            payment_date=PbDatetimeConverter().to_datetime(message.payment_date) if message.HasField('payment_date') else None,
            has_facture=message.has_facture
        )

    def to_proto(self) -> invoices_pb2.Invoice:
        return invoices_pb2.Invoice(
            invoice_id=self.invoice_id,
            eid=self.eid,
            contract_id=self.contract_id,
            contract_eid=self.contract_eid,
            amount=decimal_to_string(self.amount),
            currency=self.currency,
            status=invoice_status_converter.reversed(self.status),
            date=PbDatetimeConverter.from_datetime(self.date),
            payment_date=PbDatetimeConverter.from_datetime(self.payment_date) if self.payment_date else None,
            has_facture=self.has_facture
        )


@dataclass
class ListInvoicesInput:
    agency_id: int
    contract_id: Optional[int] = None
    date_from: Optional[datetime] = None
    date_to: Optional[datetime] = None
    status: Optional[InvoiceStatus] = None
    limit: Optional[int] = None
    offset: Optional[int] = None
    search_query: Optional[str] = None

    @classmethod
    def from_proto(cls, message: invoices_pb2.ListInvoicesInput) -> 'ListInvoicesInput':
        return cls(
            agency_id=message.agency_id,
            contract_id=message.contract_id if message.HasField('contract_id') else None,
            date_from=PbDatetimeConverter().to_datetime(message.date_from) if message.HasField('date_from') else None,
            date_to=PbDatetimeConverter().to_datetime(message.date_to) if message.HasField('date_to') else None,
            status=invoice_status_converter.forward(message.status) if message.status else None,
            limit=message.limit if message.HasField('limit') else None,
            offset=message.offset if message.HasField('offset') else None,
            search_query=message.search_query if message.HasField('search_query') else None,
        )

    def to_proto(self) -> invoices_pb2.ListInvoicesInput:
        return invoices_pb2.ListInvoicesInput(
            agency_id=self.agency_id,
            contract_id=self.contract_id,
            date_from=PbDatetimeConverter().from_datetime(self.date_from) if self.date_from else None,
            date_to=PbDatetimeConverter().from_datetime(self.date_to) if self.date_to else None,
            status=invoice_status_converter.reversed(self.status) if self.status else None,
            limit=self.limit,
            offset=self.offset,
            search_query=self.search_query,
        )


@dataclass
class Facture:
    facture_id: int
    date: datetime
    amount: Decimal
    amount_with_nds: Decimal
    nds: Decimal
    currency: str

    @classmethod
    def from_proto(cls, message: invoices_pb2.Facture) -> 'Facture':
        return cls(
            facture_id=message.facture_id,
            date=PbDatetimeConverter().to_datetime(message.date),
            amount=Decimal(message.amount),
            amount_with_nds=Decimal(message.amount_with_nds),
            nds=Decimal(message.nds),
            currency=message.currency
        )

    def to_proto(self) -> invoices_pb2.Facture:
        return invoices_pb2.Facture(
            facture_id=self.facture_id,
            date=PbDatetimeConverter().from_datetime(self.date),
            amount=decimal_to_string(self.amount),
            amount_with_nds=decimal_to_string(self.amount_with_nds),
            nds=decimal_to_string(self.nds),
            currency=self.currency
        )


@dataclass
class GetInvoiceInfoInput:
    agency_id: int
    invoice_id: int

    @classmethod
    def from_proto(cls, message: invoices_pb2.GetInvoiceInfoInput) -> 'GetInvoiceInfoInput':
        return cls(
            agency_id=message.agency_id,
            invoice_id=message.invoice_id
        )

    def to_proto(self) -> invoices_pb2.GetInvoiceInfoInput:
        return invoices_pb2.ListInvoicesInput(
            agency_id=self.agency_id,
        )


@dataclass
class DetailedInvoice:
    invoice_id: int
    eid: str
    contract_id: int
    contract_eid: str
    amount: Decimal
    currency: str
    status: InvoiceStatus
    date: datetime
    payment_date: datetime
    facture: typing.Optional[Facture] = None

    @classmethod
    def from_proto(cls, message: invoices_pb2.DetailedInvoice) -> 'DetailedInvoice':
        return DetailedInvoice(
            invoice_id=message.invoice_id,
            eid=message.eid,
            contract_id=message.contract_id,
            contract_eid=message.contract_eid,
            amount=Decimal(message.amount),
            currency=message.currency,
            status=invoice_status_converter.forward(message.status),
            date=PbDatetimeConverter().to_datetime(message.date),
            payment_date=PbDatetimeConverter().to_datetime(message.payment_date) if message.HasField('payment_date') else None,
            facture=Facture.from_proto(message.facture) if message.HasField('facture') else None
        )

    def to_proto(self) -> invoices_pb2.DetailedInvoice:
        return invoices_pb2.DetailedInvoice(
            invoice_id=self.invoice_id,
            eid=self.eid,
            contract_id=self.contract_id,
            contract_eid=self.contract_eid,
            amount=decimal_to_string(self.amount),
            currency=self.currency,
            status=invoice_status_converter.reversed(self.status),
            date=PbDatetimeConverter.from_datetime(self.date),
            payment_date=PbDatetimeConverter.from_datetime(self.payment_date) if self.payment_date else None,
            facture=self.facture.to_proto() if self.facture else None
        )


@dataclass
class GetInvoiceUrlInput:
    agency_id: int
    invoice_id: int

    @classmethod
    def from_proto(cls, message: invoices_pb2.GetInvoiceUrlInput) -> 'GetInvoiceUrlInput':
        return cls(
            agency_id=message.agency_id,
            invoice_id=message.invoice_id
        )

    def to_proto(self) -> invoices_pb2.GetInvoiceUrlInput:
        return invoices_pb2.GetInvoiceUrlInput(
            agency_id=self.agency_id,
            invoice_id=self.invoice_id
        )


@dataclass
class GetFactureUrlInput:
    agency_id: int
    facture_id: int

    @classmethod
    def from_proto(cls, message: invoices_pb2.GetFactureUrlInput) -> 'GetFactureUrlInput':
        return cls(
            agency_id=message.agency_id,
            facture_id=message.facture_id
        )

    def to_proto(self) -> invoices_pb2.GetFactureUrlInput:
        return invoices_pb2.GetFactureUrlInput(
            agency_id=self.agency_id,
            facture_id=self.facture_id
        )
