from datetime import datetime
from decimal import Decimal
from typing import Optional

from dataclasses import dataclass
from smb.common.helpers import PbDatetimeConverter

from crm.agency_cabinet.common.proto_utils import decimal_to_string
from crm.agency_cabinet.documents.proto import payments_pb2


@dataclass
class Payment:
    payment_id: int
    eid: str
    invoice_id: int
    currency: str
    amount: Decimal
    invoice_eid: str
    contract_id: int
    date: datetime

    @classmethod
    def from_proto(cls, message: payments_pb2.Payment) -> 'Payment':
        return cls(
            payment_id=message.payment_id,
            eid=message.eid,
            invoice_id=message.invoice_id,
            currency=message.currency,
            amount=Decimal(message.amount),
            invoice_eid=message.invoice_eid,
            contract_id=message.contract_id,
            date=PbDatetimeConverter().to_datetime(message.date)
        )

    def to_proto(self) -> payments_pb2.Payment:
        return payments_pb2.Payment(
            payment_id=self.payment_id,
            eid=self.eid,
            invoice_id=self.invoice_id,
            currency=self.currency,
            amount=decimal_to_string(self.amount),
            invoice_eid=self.invoice_eid,
            contract_id=self.contract_id,
            date=PbDatetimeConverter().from_datetime(self.date),
        )


@dataclass
class ListPaymentsInput:
    agency_id: int
    contract_id: Optional[int] = None
    invoice_id: Optional[int] = None
    limit: Optional[int] = None
    offset: Optional[int] = None
    date_from: Optional[datetime] = None
    date_to: Optional[datetime] = None

    @classmethod
    def from_proto(cls, message: payments_pb2.ListPaymentsInput) -> 'ListPaymentsInput':
        return cls(
            agency_id=message.agency_id,
            contract_id=message.contract_id if message.HasField('contract_id') else None,
            invoice_id=message.invoice_id if message.HasField('invoice_id') else None,
            limit=message.limit if message.HasField('limit') else None,
            offset=message.offset if message.HasField('offset') else None,
            date_from=PbDatetimeConverter().to_datetime(message.date_from) if message.HasField('date_from') else None,
            date_to=PbDatetimeConverter().to_datetime(message.date_to) if message.HasField('date_to') else None,
        )

    def to_proto(self) -> payments_pb2.ListPaymentsInput:
        return payments_pb2.ListPaymentsInput(
            agency_id=self.agency_id,
            contract_id=self.contract_id,
            invoice_id=self.invoice_id,
            limit=self.limit,
            offset=self.offset,
            date_from=PbDatetimeConverter().from_datetime(self.date_from) if self.date_from else None,
            date_to=PbDatetimeConverter().from_datetime(self.date_to) if self.date_to else None,
        )
