import datetime

from yt.wrapper import YtClient
from yql.api.v1.client import YqlClient

from crm.agency_cabinet.common.celery.base import async_to_sync
from crm.agency_cabinet.documents.server.src.config import YT_CONFIG, YQL_CONFIG
from crm.agency_cabinet.documents.server.src.db import models, db
from crm.agency_cabinet.documents.server.src.celery.base import celery_app as celery
from .load_invoices.loader import InvoicesLoader
from .load_acts.loader import ActsLoader
from .load_payments.loader import PaymentsLoader
from .load_contracts.loader import ContractsLoader
from .load_factures.loader import FacturesLoader


async def find_latest_loaded_dates(model):
    s_date = datetime.datetime(2017, 5, 1)  # earliest act available
    e_date = datetime.datetime(2017, 6, 1)

    latest_uploaded_date = await db.select([db.func.max(model.date)]).select_from(model).gino.first()

    if latest_uploaded_date[0]:
        days = 28 if model in (models.Act, models.Facture) else 33
        s_date = latest_uploaded_date[0]
        e_date = (s_date + datetime.timedelta(days=days)).replace(day=1)
    return s_date.strftime('%Y-%m-%d'), e_date.strftime('%Y-%m-%d')


@celery.task(bind=True)
def load_contracts_task(self):
    yt_client = YtClient(proxy='hahn', token=YT_CONFIG['TOKEN'])
    yql_client = YqlClient(token=YQL_CONFIG['TOKEN'])
    table_path = '//home/balance/prod/contracts/contracts_json'

    @async_to_sync
    async def load():
        await ContractsLoader(yt_client, yql_client, table_path).load()

    load()


@celery.task(bind=True)
def load_invoices_task(self, start=None, end=None):
    yt_client = YtClient(proxy='hahn', token=YT_CONFIG['TOKEN'])
    yql_client = YqlClient(token=YQL_CONFIG['TOKEN'])
    table_path = '//home/balance/prod/invoices/t_invoice'

    @async_to_sync
    async def load(start_date, end_date):
        if start_date is None or end_date is None:
            start_date, end_date = await find_latest_loaded_dates(models.Invoice)
        await InvoicesLoader(yt_client, yql_client, table_path, start_date, end_date).load()

    load(start, end)


@celery.task(bind=True)
def load_factures_task(self, date=None):  # 2020-11-01 first table with factures
    yt_client = YtClient(proxy='hahn', token=YT_CONFIG['TOKEN'])
    yql_client = YqlClient(token=YQL_CONFIG['TOKEN'])
    table_path = '//home/balance/prod/acts/f_sales_daily'

    @async_to_sync
    async def load(loading_date):
        if loading_date is None:
            _, loading_date = await find_latest_loaded_dates(models.Facture)
        await FacturesLoader(yt_client, yql_client, table_path, loading_date).load()

    load(date)


@celery.task(bind=True)
def load_acts_task(self, date=None):
    yt_client = YtClient(proxy='hahn', token=YT_CONFIG['TOKEN'])
    yql_client = YqlClient(token=YQL_CONFIG['TOKEN'])
    table_path = '//home/balance/prod/acts/f_sales_daily/'

    @async_to_sync
    async def load(loading_date):
        if loading_date is None:
            _, loading_date = await find_latest_loaded_dates(models.Act)
        await ActsLoader(yt_client, yql_client, table_path, loading_date).load()

    load(date)


@celery.task(bind=True)
def load_payments_task(self, start=None, end=None):
    yt_client = YtClient(proxy='hahn', token=YT_CONFIG['TOKEN'])
    yql_client = YqlClient(token=YQL_CONFIG['TOKEN'])
    table_path = '//home/balance/prod/bo/t_oebs_cash_payment_fact'

    @async_to_sync
    async def load(start_date, end_date):
        if start_date is None or end_date is None:
            start_date, end_date = await find_latest_loaded_dates(models.Payment)
        await PaymentsLoader(yt_client, yql_client, table_path, start_date, end_date).load()

    load(start, end)
