import logging
from datetime import datetime
from collections import defaultdict
import yt.wrapper as yt

from crm.agency_cabinet.documents.common.structs import ContractStatus
from crm.agency_cabinet.common.consts import PaymentType, DISCOUNT_TYPE_SERVICE_MAP, Services, START_FIN_YEAR_2021
from crm.agency_cabinet.common.yt.synchronizers import BaseSynchronizer
from crm.agency_cabinet.documents.server.src.db import models

LOGGER = logging.getLogger('celery.tasks.documents.contracts.synchronizers')


class ContractsSynchronizer(BaseSynchronizer):
    COLLATERAL_TYPE_ID_PROLONGATION = 80
    COLLATERAL_TYPE_ID_SERVICES = 1001
    COLLATERAL_TYPE_ID_CREDIT_LIMIT = 1004
    COLLATERAL_TYPE_ID_CREDIT_LIMIT_2 = 1033  # TODO узнать, в чём разница между 1004 и 1033

    COLLATERAL_TYPE_TO_NAME_MAP = {
        COLLATERAL_TYPE_ID_PROLONGATION: 'Продление договора',
        COLLATERAL_TYPE_ID_SERVICES: 'Добавление сервисов',
        COLLATERAL_TYPE_ID_CREDIT_LIMIT: 'Изменение кредитного лимита',
    }

    PAYMENT_TYPE_MAP = {
        2: PaymentType.prepayment.value,
        3: PaymentType.postpayment.value
    }

    agreements = defaultdict(list)

    async def process_data(self, rows: list[tuple], *args, **kwargs) -> bool:  # row: id, agency_id, eid, collaterals, inn
        for row in rows:
            contract_duplicated = await self._find_contract_duplicate(row[2])
            if contract_duplicated:
                continue

            agreements = self._parse_agreements(row[3])
            if not agreements:
                continue

            finish_date = self._extract_finish_date(row[0], agreements)
            if not finish_date or finish_date <= START_FIN_YEAR_2021:
                continue

            self.agreements[row[0]] = agreements

            payment_type = self._extract_payment_type(row[0])
            signing_date = self._extract_signing_date(row[0])
            finish_date = self._extract_finish_date(row[0])
            credit_limit = self._extract_credit_limit(row[0])
            services = self._extract_services(row[0])
            status = self._extract_status(signing_date)
            inn = row[4] or 'unknown'  # TODO nullable inn?

            await models.Contract.create(
                id=row[0],
                agency_id=row[1],
                eid=row[2],
                payment_type=payment_type,
                signing_date=signing_date,
                finish_date=finish_date,
                credit_limit=credit_limit,
                services=services,
                status=status,
                inn=inn
            )

        for contract_id, agreements in self.agreements.items():
            # первый пропускаем потому что это сам контракт (но это не точно)
            for agreement in agreements[1:]:
                if self._is_agreement_fictitious(agreement):
                    continue

                try:
                    agreement_duplicated = await self._find_agreement_duplicate(agreement.get('id'))

                    name = self._make_agreement_name(int(agreement.get('collateral_type_id')))
                    date = self._str_to_date(agreement.get('dt'))
                    got_scan = agreement.get('is_faxed') is not None
                    got_original = bool(agreement.get('sent_dt', None))

                    if agreement_duplicated:
                        await self._update_duplicate(agreement_duplicated, contract_id, name, date, got_scan, got_original)

                    else:
                        await self._create_agreement(contract_id, agreement.get('id'), name, date, got_scan,
                                                     got_original)

                except Exception:
                    LOGGER.error("Couldn't process agreement %s, contract_id=%s", agreement.get('id'), contract_id)

        return True

    def _parse_agreements(self, collaterals) -> list[dict]:
        agreements = yt.yson.yson_to_json(collaterals)
        return [agreements[i] for i in sorted(agreements.keys(), key=int)]

    def _is_agreement_fictitious(self, agreement) -> bool:
        return (
            'num' in agreement and
            isinstance(agreement['num'], str) and
            len(agreement['num']) > 0 and
            agreement['num'][0] == 'Ф'
        )

    def _extract_finish_date(self, contract_id, agreements=None):
        agreement = self._get_last_agreement_with_type(contract_id, self.COLLATERAL_TYPE_ID_PROLONGATION, agreements)

        if not agreement:
            agreement = self._get_first_agreement(contract_id, agreements)

        finish_date = None
        if 'finish_dt' in agreement:
            finish_date = agreement.get('finish_dt')
        elif 'end_dt' in agreement:  # TODO возможно end_dt значит другое?
            finish_date = agreement.get('end_dt')
        return self._str_to_date(finish_date)

    def _get_last_agreement_with_type(self, contract_id: int, type: int, agreements: list[dict] = None) -> dict:
        if not agreements:
            agreements = self.agreements[contract_id]
        return next((agr for agr in reversed(agreements) if agr.get('collateral_type_id') == type), None)

    def _extract_signing_date(self, contract_id):
        agreement = self._get_first_agreement(contract_id)
        signing_date = agreement.get('is_signed')
        return self._str_to_date(signing_date)

    def _extract_payment_type(self, contract_id) -> PaymentType:
        agreement = self._get_first_agreement(contract_id)
        payment_type = agreement.get('payment_type')
        return self.PAYMENT_TYPE_MAP.get(payment_type, None)

    def _extract_credit_limit(self, contract_id) -> int:
        agreement = self._get_last_agreement_with_type(contract_id, self.COLLATERAL_TYPE_ID_CREDIT_LIMIT)

        if not agreement:
            agreement = self._get_last_agreement_with_type(contract_id, self.COLLATERAL_TYPE_ID_CREDIT_LIMIT_2)

        if not agreement:
            agreement = self._get_first_agreement(contract_id)

        return agreement.get('credit_limit_single', 0)

    def _str_to_date(self, date: str) -> datetime:
        return datetime.strptime(date, '%Y-%m-%dT%H:%M:%S') if date else None

    def _get_first_agreement(self, contract_id, agreements=None) -> dict:
        if agreements:
            return agreements[0]
        return self.agreements[contract_id][0]

    def _extract_services(self, contract_id) -> list[Services]:
        agreement = self._get_last_agreement_with_type(contract_id, self.COLLATERAL_TYPE_ID_SERVICES)
        if not agreement:
            agreement = self._get_first_agreement(contract_id)

        services = []

        if agreement.get('services'):
            for service, enabled in agreement.get('services').items():
                if enabled == 1 and int(service) in DISCOUNT_TYPE_SERVICE_MAP:
                    services.append(DISCOUNT_TYPE_SERVICE_MAP[int(service)])
        return services

    def _make_agreement_name(self, type: int) -> str:
        return self.COLLATERAL_TYPE_TO_NAME_MAP.get(type, f'{type}')

    def _extract_status(self, signing_date) -> ContractStatus:
        return ContractStatus.valid.value if signing_date is not None else ContractStatus.not_signed.value

    async def _find_agreement_duplicate(self, agreement_id):
        return await models.Agreement.query.where(models.Agreement.id == agreement_id).gino.first()

    async def _find_contract_duplicate(self, contract_eid):
        return await models.Contract.query.where(models.Contract.eid == contract_eid).gino.first()

    async def _update_duplicate(self, agreement, contract_id, name, date, got_scan, got_original):
        await agreement.update(
            contract_id=contract_id,
            name=name,
            date=date,
            got_scan=got_scan,
            got_original=got_original
        ).apply()

    async def _create_agreement(self, contract_id, agreement_id, name, date, got_scan, got_original):
        await models.Agreement.create(
            id=agreement_id,
            contract_id=contract_id,
            name=name,
            date=date,
            got_scan=got_scan,
            got_original=got_original
        )
