import typing
from library.python.monlib.metric_registry import MetricRegistry

from crm.agency_cabinet.documents.proto import request_pb2, common_pb2, contracts_pb2, invoices_pb2, payments_pb2, \
    acts_pb2, agreements_pb2
from crm.agency_cabinet.documents.server.src import procedures
from smb.common.rmq.rpc.server import BaseRpcHandler
from crm.agency_cabinet.documents.common import structs
from crm.agency_cabinet.documents.server.src.exceptions import NoSuchContractException, UnsuitableAgencyException, \
    NoSuchInvoiceException, NoSuchActException, FileNotFound, NoSuchFactureException, NoSuchAgreementException
from crm.agency_cabinet.common.yadoc import YaDocClient
from crm.agency_cabinet.common.server.common.tvm import get_tvm_client
from crm.agency_cabinet.common.server.common.tvm import Tvm2Config
from crm.agency_cabinet.common.consts.tvm import TVMIdTest


class Handler(BaseRpcHandler):
    _request_proto = request_pb2.RpcRequest

    def __init__(
        self,
        yadoc_endpoint_url: str = None,
        yadoc_tvm_id: int = None,
        tvm_config: Tvm2Config = None,
        metric_registry: MetricRegistry = None,
    ):
        self.metric_registry = metric_registry
        self.yadoc_endpoint_url = yadoc_endpoint_url
        self.tvm_config = tvm_config
        self.yadoc_tvm_id = yadoc_tvm_id if yadoc_tvm_id is not None else TVMIdTest.yadoc.value
        self.yadoc_client: typing.Optional[YaDocClient] = None

    async def _setup_tvm_client(self) -> bool:
        if self.tvm_config:
            self.tvm_client = get_tvm_client(self.tvm_config)
            return True
        return False

    async def _setup_yadoc(self) -> bool:
        if self.yadoc_endpoint_url:
            self.yadoc_client = YaDocClient(
                self.yadoc_endpoint_url,
                tvm_client=self.tvm_client,
                yadoc_tvm_id=self.yadoc_tvm_id,
                raise_for_status=True
            )
            return True
        return False

    async def setup(self) -> bool:
        success = await self._setup_tvm_client() and await self._setup_yadoc()
        return success

    async def teardown(self):
        if self.yadoc_client is not None:
            await self.yadoc_client.close_session()

    async def ping(self, _: common_pb2.Empty) -> common_pb2.PingOutput:
        return common_pb2.PingOutput(ping='pong')

    async def list_contracts(
        self, message: contracts_pb2.ListContractsInput
    ) -> contracts_pb2.ListContractsOutput:
        params = structs.ListContractsInput.from_proto(message)

        contracts = await procedures.ListContracts()(params=params)

        return contracts_pb2.ListContractsOutput(
            contracts=contracts_pb2.ContractsList(
                contracts=[contract.to_proto() for contract in contracts]
            )
        )

    async def get_contract_info(
        self, message: contracts_pb2.GetContractInfoInput
    ) -> contracts_pb2.GetContractInfoOutput:
        try:
            params = structs.GetContractInfoInput.from_proto(message)
            contract = await procedures.GetContractInfo()(params=params)

            return contracts_pb2.GetContractInfoOutput(contract=contract.to_proto())
        except NoSuchContractException:
            return contracts_pb2.GetContractInfoOutput(no_such_contract=common_pb2.Empty())
        except UnsuitableAgencyException:
            return contracts_pb2.GetContractInfoOutput(unsuitable_agency=common_pb2.Empty())

    async def list_invoices(
        self, message: invoices_pb2.ListInvoicesInput
    ) -> invoices_pb2.ListInvoicesOutput:
        params = structs.ListInvoicesInput.from_proto(message)

        invoices = await procedures.ListInvoices()(params=params)

        return invoices_pb2.ListInvoicesOutput(
            invoices=invoices_pb2.InvoicesList(
                invoices=[invoice.to_proto() for invoice in invoices]))

    async def list_payments(
        self, message: payments_pb2.ListPaymentsInput
    ) -> payments_pb2.ListPaymentsOutput:
        params = structs.ListPaymentsInput.from_proto(message)

        payments = await procedures.ListPayments()(params=params)

        return payments_pb2.ListPaymentsOutput(
            payments=payments_pb2.PaymentsList(
                payments=[payment.to_proto() for payment in payments]
            )
        )

    async def list_acts(
        self, message: acts_pb2.ListActsInput
    ) -> acts_pb2.ListActsOutput:
        params = structs.ListActsInput.from_proto(message)

        acts = await procedures.ListActs()(params=params)

        return acts_pb2.ListActsOutput(
            acts=acts_pb2.ActsList(
                acts=[act.to_proto() for act in acts]
            )
        )

    async def get_invoice_info(
        self, message: invoices_pb2.GetInvoiceInfoInput
    ) -> invoices_pb2.GetInvoiceInfoOutput:
        try:
            params = structs.GetInvoiceInfoInput.from_proto(message)
            invoice = await procedures.GetInvoiceInfo()(params=params)

            return invoices_pb2.GetInvoiceInfoOutput(invoice=invoice.to_proto())
        except NoSuchInvoiceException:
            return invoices_pb2.GetInvoiceInfoOutput(no_such_invoice=common_pb2.Empty())
        except UnsuitableAgencyException:
            return invoices_pb2.GetInvoiceInfoOutput(unsuitable_agency=common_pb2.Empty())

    async def list_agreements(
        self, message: agreements_pb2.ListAgreementsInput
    ) -> agreements_pb2.ListAgreementsOutput:
        params = structs.ListAgreementsInput.from_proto(message)

        agreements = await procedures.ListAgreements()(params=params)

        return agreements_pb2.ListAgreementsOutput(
            agreements=agreements_pb2.AgreementsList(
                agreements=[agreement.to_proto() for agreement in agreements]
            )
        )

    async def get_invoice_url(
        self, message: invoices_pb2.GetInvoiceUrlInput
    ) -> invoices_pb2.GetInvoiceUrlOutput:
        try:
            params = structs.GetInvoiceUrlInput.from_proto(message)
            url = await procedures.GetInvoiceUrl()(params=params, yadoc_client=self.yadoc_client)

            return invoices_pb2.GetInvoiceUrlOutput(result=url.to_proto())
        except NoSuchInvoiceException:
            return invoices_pb2.GetInvoiceUrlOutput(no_such_invoice=common_pb2.Empty())
        except UnsuitableAgencyException:
            return invoices_pb2.GetInvoiceUrlOutput(unsuitable_agency=common_pb2.Empty())
        except FileNotFound:
            return invoices_pb2.GetInvoiceUrlOutput(file_not_found=common_pb2.Empty())

    async def get_act_url(
        self, message: acts_pb2.GetActUrlInput
    ) -> acts_pb2.GetActUrlOutput:
        try:
            params = structs.GetActUrlInput.from_proto(message)
            url = await procedures.GetActUrl()(params=params, yadoc_client=self.yadoc_client)

            return acts_pb2.GetActUrlOutput(result=url.to_proto())
        except NoSuchActException:
            return acts_pb2.GetActUrlOutput(no_such_act=common_pb2.Empty())
        except UnsuitableAgencyException:
            return acts_pb2.GetActUrlOutput(unsuitable_agency=common_pb2.Empty())
        except FileNotFound:
            return acts_pb2.GetActUrlOutput(file_not_found=common_pb2.Empty())

    async def get_facture_url(
        self, message: invoices_pb2.GetFactureUrlInput
    ) -> invoices_pb2.GetFactureUrlOutput:
        try:
            params = structs.GetFactureUrlInput.from_proto(message)
            url = await procedures.GetFactureUrl()(params=params, yadoc_client=self.yadoc_client)

            return invoices_pb2.GetFactureUrlOutput(result=url.to_proto())
        except NoSuchFactureException:
            return invoices_pb2.GetFactureUrlOutput(no_such_facture=common_pb2.Empty())
        except UnsuitableAgencyException:
            return invoices_pb2.GetFactureUrlOutput(unsuitable_agency=common_pb2.Empty())
        except FileNotFound:
            return invoices_pb2.GetFactureUrlOutput(file_not_found=common_pb2.Empty())

    async def get_contract_url(
        self, message: contracts_pb2.GetContractUrlInput
    ) -> contracts_pb2.GetContractUrlOutput:
        try:
            params = structs.GetContractUrlInput.from_proto(message)
            url = await procedures.GetContractUrl()(params=params)

            return contracts_pb2.GetContractUrlOutput(result=url.to_proto())
        except NoSuchContractException:
            return contracts_pb2.GetContractUrlOutput(no_such_contract=common_pb2.Empty())
        except UnsuitableAgencyException:
            return contracts_pb2.GetContractUrlOutput(unsuitable_agency=common_pb2.Empty())
        except FileNotFound:
            return contracts_pb2.GetContractUrlOutput(file_not_found=common_pb2.Empty())

    async def get_agreement_url(
        self, message: agreements_pb2.GetAgreementUrlInput
    ) -> agreements_pb2.GetAgreementUrlOutput:
        try:
            params = structs.GetAgreementUrlInput.from_proto(message)
            url = await procedures.GetAgreementUrl()(params=params)

            return agreements_pb2.GetAgreementUrlOutput(result=url.to_proto())
        except NoSuchAgreementException:
            return agreements_pb2.GetAgreementUrlOutput(no_such_agreement=common_pb2.Empty())
        except UnsuitableAgencyException:
            return agreements_pb2.GetAgreementUrlOutput(unsuitable_agency=common_pb2.Empty())
        except FileNotFound:
            return agreements_pb2.GetAgreementUrlOutput(file_not_found=common_pb2.Empty())
