import datetime
from dataclasses import dataclass
from aiohttp import ClientResponseError
from aiohttp.web import HTTPNotFound
from crm.agency_cabinet.common.yadoc import YaDocClient, YaDocType
from crm.agency_cabinet.common.server.common.structs.common import UrlResponse
from crm.agency_cabinet.documents.server.src.db import (
    Act, Contract, Invoice, db
)
from sqlalchemy import or_
from crm.agency_cabinet.documents.server.src.exceptions import UnsuitableAgencyException, NoSuchActException, FileNotFound
from crm.agency_cabinet.documents.common import structs
from crm.agency_cabinet.documents.server.src.db.queries import update_yadoc_id_and_get_yadoc_url


@dataclass
class ListActs:
    async def __call__(self, params: structs.ListActsInput) -> [structs.Act]:
        query = db.select(
            [
                Act.id,
                Act.amount,
                Act.currency,
                Act.date,
                Act.eid,
                Invoice.id.label('invoice_id'),
                Invoice.contract_id,
                Invoice.eid.label('invoice_eid'),
                Contract.eid.label('contract_eid'),
            ]
        ).select_from(
            Act.join(Invoice).join(Contract)
        ).where(
            Contract.agency_id == params.agency_id
        )

        if params.search_query:
            search_query = '%{}%'.format(params.search_query.lower())

            query = query.where(
                or_(
                    db.func.lower(Act.eid).like(search_query),
                    db.func.lower(Invoice.eid).like(search_query),
                    db.func.lower(Contract.eid).like(search_query),
                )
            )

        if params.contract_id:
            query = query.where(
                Invoice.contract_id == params.contract_id
            )

        if params.invoice_id:
            query = query.where(
                Invoice.id == params.invoice_id
            )

        if params.date_from:
            query = query.where(
                Act.date >= params.date_from
            )

        if params.date_to:
            query = query.where(
                Act.date < params.date_to
            )

        if params.limit:
            query = query.limit(params.limit)

        if params.offset:
            query = query.offset(params.offset)

        acts = await query.order_by(Act.date.desc()).gino.all()

        return [
            structs.Act(
                act_id=act.id,
                invoice_id=act.invoice_id,
                amount=act.amount,
                eid=act.eid,
                currency=act.currency,
                date=act.date,
                contract_id=act.contract_id,
                contract_eid=act.contract_eid,
                invoice_eid=act.invoice_eid,
            ) for act in acts
        ]


@dataclass
class GetActUrl:
    async def __call__(self, params: structs.GetActUrlInput, yadoc_client: YaDocClient) -> UrlResponse:
        act = await db.select(
            [
                Contract.agency_id,
                Act.id,
                Act.eid,
                Act.date,
                Act.amount,
                Act.yadoc_id,
                Invoice.contract_id.label('contract_id'),
                Invoice.eid.label('bill_number')

            ]
        ).select_from(
            Act.join(Invoice).join(Contract)
        ).where(
            Act.id == params.act_id
        ).gino.first()

        if act is None:
            raise NoSuchActException()

        if act.agency_id != params.agency_id:
            raise UnsuitableAgencyException()

        try:
            if act.yadoc_id is not None:
                url = await yadoc_client.get_doc_url(act.yadoc_id)
            else:
                doc_info = await yadoc_client.get_first_doc_info(
                    date_from=act.date,
                    date_to=act.date + datetime.timedelta(days=1),
                    contract_ids=[act.contract_id],
                    bill_numbers=[act.bill_number],
                    doc_types=[YaDocType.act.value],
                )
                url = await update_yadoc_id_and_get_yadoc_url(doc_info, yadoc_client, Act, act.id)
            return UrlResponse(url=url)
        except ClientResponseError as ex:
            if ex.status == HTTPNotFound.status_code:
                raise FileNotFound from ex
            raise ex
