from dataclasses import dataclass
from sqlalchemy import and_, or_
from aiohttp import ClientResponseError
from aiohttp.web import HTTPNotFound
from crm.agency_cabinet.common.server.common.structs.common import UrlResponse
from crm.agency_cabinet.documents.common.structs import (
    ListAgreementsInput,
    Agreement,
    GetAgreementUrlInput
)
from crm.agency_cabinet.documents.server.src.db import models, db
from crm.agency_cabinet.documents.server.src.exceptions import NoSuchAgreementException, UnsuitableAgencyException, FileNotFound


@dataclass
class ListAgreements:
    async def __call__(self, params: ListAgreementsInput) -> [Agreement]:
        query = db.select(
            [
                models.Agreement.id,
                models.Agreement.name,
                models.Agreement.date,
                models.Agreement.got_scan,
                models.Agreement.got_original
            ]

        ).select_from(
            models.Contract.outerjoin(models.Agreement)
        ).where(
            and_(
                models.Contract.agency_id == params.agency_id,
                models.Agreement.contract_id == params.contract_id
            )
        )

        if params.search_query:
            search_query = '%{}%'.format(params.search_query.lower())

            query = query.where(
                or_(
                    db.func.lower(models.Contract.eid).like(search_query),
                    db.func.lower(models.Agreement.name).like(search_query),
                )
            )

        if params.date_from:
            query = query.where(
                models.Agreement.date >= params.date_from
            )

        if params.date_to:
            query = query.where(
                models.Agreement.date < params.date_to
            )

        if params.limit:
            query = query.limit(params.limit)

        if params.offset:
            query = query.offset(params.offset)

        agreements = await query.order_by(models.Agreement.date.desc()).gino.all()

        return [
            Agreement(
                agreement_id=agreement.id,
                name=agreement.name,
                got_scan=bool(agreement.got_scan),
                got_original=bool(agreement.got_original),
                date=agreement.date
            ) for agreement in agreements
        ]


@dataclass
class GetAgreementUrl:
    async def __call__(self, params: GetAgreementUrlInput) -> UrlResponse:
        agreement = await db.select(
            [
                models.Agreement.id,
                models.Contract.id.label('contract_id'),
                models.Contract.agency_id.label('agency_id')
            ]

        ).select_from(
            models.Contract.outerjoin(models.Agreement)
        ).where(
            models.Agreement.id == params.agreement_id
        ).gino.first()

        if agreement is None:
            raise NoSuchAgreementException()

        if agreement.agency_id != params.agency_id:
            raise UnsuitableAgencyException()

        try:
            raise FileNotFound()
            # return UrlResponse(url=url)
        except ClientResponseError as ex:
            if ex.status == HTTPNotFound.status_code:
                raise FileNotFound from ex
            raise ex
