from dataclasses import dataclass
from aiohttp import ClientResponseError
from aiohttp.web import HTTPNotFound
from crm.agency_cabinet.common.consts import PaymentType, Services
from crm.agency_cabinet.documents.common.structs import (
    ListContractsInput,
    Contract,
    ContractStatus,
    GetContractInfoInput,
    GetContractUrlInput
)
from crm.agency_cabinet.common.server.common.structs.common import UrlResponse
from crm.agency_cabinet.documents.server.src.db import models, db
from crm.agency_cabinet.documents.server.src.exceptions import NoSuchContractException, UnsuitableAgencyException, FileNotFound


@dataclass
class ListContracts:
    async def __call__(self, params: ListContractsInput) -> [Contract]:
        query = db.select(
            [
                models.Contract.id,
                models.Contract.eid,
                models.Contract.inn,
                models.Contract.payment_type,
                models.Contract.status,
                models.Contract.services,
                models.Contract.signing_date,
                models.Contract.finish_date,
                models.Contract.credit_limit

            ]
        ).select_from(
            models.Contract
        ).where(
            models.Contract.agency_id == params.agency_id
        )

        contracts = await query.order_by(models.Contract.finish_date.desc(), models.Contract.id).gino.all()

        return [
            Contract(
                contract_id=contract.id,
                eid=contract.eid,
                payment_type=PaymentType(contract.payment_type) if contract.payment_type else None,
                status=ContractStatus(contract.status),
                services=[Services(service) for service in (set(contract.services) if contract.services else {})],
                signing_date=contract.signing_date,
                finish_date=contract.finish_date,
            ) for contract in contracts
        ]


@dataclass
class GetContractInfo:
    async def __call__(self, params: GetContractInfoInput) -> Contract:
        contract = await models.Contract.query.where(
            models.Contract.id == params.contract_id
        ).gino.first()

        if contract is None:
            raise NoSuchContractException
        if contract.agency_id != params.agency_id:
            raise UnsuitableAgencyException

        return Contract(
            contract_id=contract.id,
            eid=contract.eid,
            inn=contract.inn,
            payment_type=PaymentType(contract.payment_type) if contract.payment_type else None,
            status=ContractStatus(contract.status),
            services=[Services(service) for service in (set(contract.services) if contract.services else {})],
            signing_date=contract.signing_date,
            finish_date=contract.finish_date,
            credit_limit=contract.credit_limit,
        )


@dataclass
class GetContractUrl:
    async def __call__(self, params: GetContractUrlInput) -> UrlResponse:
        contract = await models.Contract.query.where(
            models.Contract.id == params.contract_id
        ).gino.first()

        if contract is None:
            raise NoSuchContractException()

        if contract.agency_id != params.agency_id:
            raise UnsuitableAgencyException()

        try:
            raise FileNotFound()
            # return UrlResponse(url=url)
        except ClientResponseError as ex:
            if ex.status == HTTPNotFound.status_code:
                raise FileNotFound from ex
            raise ex
