import datetime
from dataclasses import dataclass
from aiohttp import ClientResponseError
from aiohttp.web import HTTPNotFound

from crm.agency_cabinet.common.yadoc import YaDocClient, YaDocType
from crm.agency_cabinet.common.server.common.structs.common import UrlResponse
from crm.agency_cabinet.documents.common import structs
from crm.agency_cabinet.documents.server.src.db import models, db
from crm.agency_cabinet.documents.server.src.exceptions import \
    UnsuitableAgencyException, NoSuchInvoiceException, FileNotFound, NoSuchFactureException
from crm.agency_cabinet.documents.server.src.db.queries import update_yadoc_id_and_get_yadoc_url


@dataclass
class ListInvoices:
    async def __call__(self, params: structs.ListInvoicesInput) -> [structs.Invoice]:
        query = db.select(
            [
                models.Contract.eid.label('contract_eid'),
                models.Invoice.id,
                models.Invoice.eid,
                models.Invoice.contract_id,
                models.Invoice.amount,
                models.Invoice.currency,
                models.Invoice.status,
                models.Invoice.date,
                models.Invoice.payment_date,
                models.Facture.id.label('facture_id')
            ]
        ).select_from(
            models.Invoice.join(models.Contract).outerjoin(models.Facture)
        ).where(
            models.Contract.agency_id == params.agency_id
        )

        if params.search_query:
            query = query.where(
                db.func.lower(models.Invoice.eid).like(f'%{params.search_query.lower()}%')
            )

        if params.contract_id:
            query = query.where(
                models.Invoice.contract_id == params.contract_id
            )

        if params.date_from:
            query = query.where(
                models.Invoice.date >= params.date_from
            )

        if params.date_to:
            query = query.where(
                models.Invoice.date < params.date_to
            )

        if params.status:
            query = query.where(
                models.Invoice.status == params.status.value
            )

        if params.limit:
            query = query.limit(params.limit)

        if params.offset:
            query = query.offset(params.offset)

        invoices = await query.order_by(models.Invoice.date.desc()).gino.all()

        return [
            structs.Invoice(
                invoice_id=invoice.id,
                eid=invoice.eid,
                contract_id=invoice.contract_id,
                contract_eid=invoice.contract_eid,
                amount=invoice.amount,
                currency=invoice.currency,
                status=structs.InvoiceStatus(invoice.status),
                date=invoice.date,
                payment_date=invoice.payment_date,
                has_facture=bool(invoice.facture_id)
            ) for invoice in invoices
        ]


@dataclass
class GetInvoiceInfo:
    async def __call__(self, params: structs.GetInvoiceInfoInput) -> structs.DetailedInvoice:
        invoice = await db.select(
            [
                models.Contract.agency_id,
                models.Contract.eid.label('contract_eid'),
                models.Invoice.id,
                models.Invoice.eid,
                models.Invoice.contract_id,
                models.Invoice.amount,
                models.Invoice.currency,
                models.Invoice.status,
                models.Invoice.date,
                models.Invoice.payment_date,
                models.Facture.id.label('facture_id'),
                models.Facture.date.label('facture_date'),
                models.Facture.amount.label('facture_amount'),
                models.Facture.amount_with_nds.label('facture_amount_with_nds'),
                models.Facture.nds.label('facture_nds'),
                models.Facture.currency.label('facture_currency'),
            ]
        ).select_from(
            models.Invoice.join(models.Contract).outerjoin(models.Facture)
        ).where(
            models.Invoice.id == params.invoice_id
        ).gino.first()

        if invoice is None:
            raise NoSuchInvoiceException

        if invoice.agency_id != params.agency_id:
            raise UnsuitableAgencyException

        result = structs.DetailedInvoice(
            invoice_id=invoice.id,
            eid=invoice.eid,
            contract_id=invoice.contract_id,
            contract_eid=invoice.contract_eid,
            amount=invoice.amount,
            currency=invoice.currency,
            status=structs.InvoiceStatus(invoice.status),
            date=invoice.date,
            payment_date=invoice.payment_date,
            facture=None
        )

        if invoice.facture_id is not None:
            result.facture = structs.Facture(
                facture_id=invoice.facture_id,
                date=invoice.facture_date,
                amount=invoice.facture_amount,
                amount_with_nds=invoice.facture_amount_with_nds,
                nds=invoice.facture_nds,
                currency=invoice.facture_currency
            )

        return result


@dataclass
class GetInvoiceUrl:
    async def __call__(self, params: structs.GetInvoiceUrlInput, yadoc_client: YaDocClient) -> UrlResponse:
        invoice = await db.select(
            [
                models.Contract.agency_id,
                models.Invoice.id,
                models.Invoice.eid.label('doc_number'),
                models.Invoice.contract_id,
                models.Invoice.date,
                models.Invoice.yadoc_id
            ]
        ).select_from(
            models.Invoice.join(models.Contract)
        ).where(
            models.Invoice.id == params.invoice_id
        ).gino.first()

        if invoice is None:
            raise NoSuchInvoiceException()

        if invoice.agency_id != params.agency_id:
            raise UnsuitableAgencyException()

        try:
            if invoice.yadoc_id is not None:
                url = await yadoc_client.get_doc_url(invoice.yadoc_id)
            else:
                doc_info = await yadoc_client.get_first_doc_info(
                    date_from=invoice.date,
                    date_to=invoice.date + datetime.timedelta(days=1),
                    contract_ids=[invoice.contract_id],
                    doc_numbers=[invoice.doc_number],
                    doc_types=[YaDocType.invoice.value],
                )
                url = await update_yadoc_id_and_get_yadoc_url(doc_info, yadoc_client, models.Invoice, invoice.id)
            return UrlResponse(url=url)
        except ClientResponseError as ex:
            if ex.status == HTTPNotFound.status_code:
                raise FileNotFound from ex
            raise ex


@dataclass
class GetFactureUrl:
    async def __call__(self, params: structs.GetFactureUrlInput, yadoc_client: YaDocClient) -> UrlResponse:
        facture = await db.select(
            [
                models.Contract.agency_id,
                models.Facture.id,
                models.Facture.yadoc_id,
                models.Invoice.eid.label('bill_number'),
                models.Invoice.contract_id,
                models.Facture.date
            ]
        ).select_from(
            models.Invoice.join(models.Contract)
        ).where(
            models.Facture.id == params.facture_id
        ).gino.first()

        if facture is None:
            raise NoSuchFactureException()

        if facture.agency_id != params.agency_id:
            raise UnsuitableAgencyException()

        try:
            if facture.yadoc_id is not None:
                url = await yadoc_client.get_doc_url(facture.yadoc_id)
            else:
                doc_info = await yadoc_client.get_first_doc_info(
                    date_from=facture.date,
                    date_to=facture.date + datetime.timedelta(days=1),
                    contract_ids=[facture.contract_id],
                    bill_numbers=[facture.bill_number],
                    doc_types=[YaDocType.facture.value],
                )
                url = await update_yadoc_id_and_get_yadoc_url(doc_info, yadoc_client, models.Facture, facture.id)
            return UrlResponse(url=url)
        except ClientResponseError as ex:
            if ex.status == HTTPNotFound.status_code:
                raise FileNotFound from ex
            raise ex
